using System;
using System.Data;
using System.Collections;
using System.Data.SqlClient;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{


	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///	<Developer>Mike Smith</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>9/17/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///Blood Unit Test business object layer class
	///</summary>

	#endregion


	public class BloodUnitTest : BloodTestBase
	{

		private BOL.BloodUnit _bloodUnit;

		private System.Guid _bloodUnitGuid = System.Guid.Empty;
		private string _testTechId;
		
		private string _entryMethodCode;
		private BOL.TestingMethod _testingMethod;
		private Guid _antiseraWorklistTypingGuid;
		private Guid _confirmationWorklistUnitGuid;
		private int _orderableTestId;
		private BOL.Reagent _reagent;
		private string _specificity;
		private BOL.TestResult _testResult;
		private Guid _orderedUnitGuid;
		private string _recordStatusCode;
		private BOL.Rack _rack;
		private string _autoInstrumentName;
        //Defect 251282 Added UnitResultSetGuid to allow differentation between tests for automated instrument tests
        //tests coming from auto instrument don't have the worklist guid assigned to them
        private Guid _unitResultSetGuid;


		private string _entryTechId = string.Empty;

		#region Constructors/Destructors

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1713"> 
		///		<ExpectedInput>BloodUnitGuid, BloodTestTypeId</ExpectedInput>
		///		<ExpectedOutput>A new BloodUnitTest object with properties set to input values</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1714"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Constructor for BloodUnitTest - BloodUnitGuid/TestTypeId are the only NotNull fields
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="bloodTestTypeId"></param>
		public BloodUnitTest(System.Guid bloodUnitGuid, int bloodTestTypeId)
		{
			this.BloodUnitTestGuid = System.Guid.NewGuid();
			this.BloodUnitGuid = bloodUnitGuid;
			this.BloodTestTypeObj = new BOL.BloodTestType(bloodTestTypeId);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="536"> 
		///		<ExpectedInput>All properties of a blood unit test</ExpectedInput>
		///		<ExpectedOutput>A new blood unit test object with properties set to input values</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1678"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Blood Unit Test class constructor. Accepts parameters to initialize property values.
		/// </summary>
		/// <param name="bloodUnitTestGuid"></param>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="bloodTestTypeId"></param>
		/// <param name="testDate"></param>
		/// <param name="testResultId"></param>
		/// <param name="testComments"></param>
		/// <param name="entryMethodCode"></param>
		/// <param name="divisionCode"></param>
		public BloodUnitTest(Guid bloodUnitTestGuid,
			Guid bloodUnitGuid,
			int bloodTestTypeId,
			DateTime testDate,
			string testResultId,
			string testComments,
			string entryMethodCode,
			string divisionCode)
		{
			this.BloodUnitTestGuid = bloodUnitTestGuid;
			this.BloodUnitGuid = bloodUnitGuid;
			this.BloodTestTypeObj = new BOL.BloodTestType(bloodTestTypeId);
			this.TestDate = testDate;
			this.TestResultId = testResultId;
			this.TestComments = testComments;
			this.EntryMethodCode = entryMethodCode;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2400"> 
		///		<ExpectedInput>DataRow containing BloodUnitTest data</ExpectedInput>
		///		<ExpectedOutput>BloodUnitTest object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1679"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates BloodUnitTest object and populates it with data from DataRow
		/// </summary>
		/// <param name="dr"></param>
		public BloodUnitTest(DataRow dr)
		{
			this.LoadFromDataRow(dr);
		}


		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="538"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="699"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets blood unit unique identifier.
		/// </summary>
		public System.Guid BloodUnitGuid
		{
			get
			{
				return this._bloodUnitGuid;
			}
			set
			{
				this._bloodUnitGuid = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="539"> 
		///		<ExpectedInput>String</ExpectedInput>
		///		<ExpectedOutput>String</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="700"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets the entry method of the blood test.
		/// </summary>
		public string EntryMethodCode
		{
			get
			{
				return this._entryMethodCode;
			}
			set
			{
				this._entryMethodCode = value;
			}
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3663"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3664"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		/// <summary>
		/// Gets or sets the antisera worklist typing Guid of the blood unit test object
		/// </summary>
		public Guid AntiseraWorklistTypingGuid
		{
			get
			{
				return this._antiseraWorklistTypingGuid;
			}
			set
			{
				this._antiseraWorklistTypingGuid = value;
			}
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3758"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3759"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets the confirmation worklist unit Guid of the blood unit test object
		/// </summary>
		public Guid ConfirmationWorklistUnitGuid
		{
			get
			{
				return this._confirmationWorklistUnitGuid;
			}
			set
			{
				this._confirmationWorklistUnitGuid = value;
			}
		}		

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3760"> 
		///		<ExpectedInput>Int</ExpectedInput>
		///		<ExpectedOutput>Int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3761"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets the orderable test Id of the blood unit test object
		/// </summary>
		public int OrderableTestId
		{
			get
			{
				return this._orderableTestId;
			}
			set
			{
				this._orderableTestId = value;
			}
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3762"> 
		///		<ExpectedInput>BOL.Reagent object</ExpectedInput>
		///		<ExpectedOutput>BOL.Reagent object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3763"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets the reagent of the blood unit test object
		/// </summary>
		public BOL.Reagent Reagent
		{
			get
			{
				return this._reagent;
			}
			set
			{
				this._reagent = value;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/17/2006</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8381"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8382"> 
		///		<ExpectedInput>Invalid Code</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// RecordStatusCode
		/// </summary>
		public string RecordStatusCode
		{
			get
			{
				
				return this._recordStatusCode;
			}
			set
			{
				if (Common.Utility.GetRecordStatusCodeFromString(value) == Common.RecordStatusCode.Unknown)
				{
					throw new ArgumentException("Invalid RecordStatusCode");
				}

				this._recordStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3764"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3765"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Specificity
		/// </summary>
		public string Specificity
		{
			get
			{
				return this._specificity;
			}
			set
			{
				this._specificity = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1680"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1681"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// TestTechId
		/// </summary>
		public string TestTechId
		{
			get
			{
				return this._testTechId;
			}
			set
			{
				this._testTechId = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1682"> 
		///		<ExpectedInput>BOL.TestResult</ExpectedInput>
		///		<ExpectedOutput>BOL.TestResult</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1683"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// TestResult
		/// </summary>
		public BOL.TestResult TestResult
		{
			get
			{
				return this._testResult;
			}
			set
			{
				this._testResult = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1684"> 
		///		<ExpectedInput>BOL.TestingMethod</ExpectedInput>
		///		<ExpectedOutput>BOL.TestingMethod</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1685"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// TestingMethod
		/// </summary>
		public BOL.TestingMethod TestingMethod
		{
			get
			{
				return this._testingMethod;
			}
			set
			{
				this._testingMethod = value;
			}
		}

		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1686"> 
		///		<ExpectedInput>BOL.BloodUnit</ExpectedInput>
		///		<ExpectedOutput>BOL.BloodUnit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1687"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// BloodUnit
		/// </summary>
		public BOL.BloodUnit BloodUnit
		{
			get
			{
				return this._bloodUnit;
			}
			set
			{
				this._bloodUnit = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/24/2004</CreationDate>
		///<TestCases>
		/// 
		///<Case type="0" testid ="7402"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7403"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// AutoInstrumentName
		/// </summary>
		public string AutoInstrumentName
		{
			get
			{
				return this._autoInstrumentName;
			}
			set
			{
				this._autoInstrumentName = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6779"> 
		///		<ExpectedInput>Rack object</ExpectedInput>
		///		<ExpectedOutput>Rack object</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6780"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Rack
		/// </summary>
		public BOL.Rack Rack
		{
			get
			{
				return this._rack;
			}
			set
			{
				this._rack = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7406"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7407"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderedUnitGuid
		/// </summary>
		public Guid OrderedUnitGuid
		{
			get
			{
				return this._orderedUnitGuid;
			}
			set
			{
				this._orderedUnitGuid = value;
			}
		}

        //Defect 251282 Added UnitResultSetGuid to allow differentation between tests for automated instrument tests
        //tests coming from auto instrument don't have the worklist guid assigned to them
        /// <summary>
        /// UnitResultSetGuid
        /// </summary>
        public Guid UnitResultSetGuid
        {
            get
            {
                return this._unitResultSetGuid;
            }
            set
            {
                this._unitResultSetGuid = value;
            }
        }
		#endregion

		#region Common Methods

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1444"> 
		///		<ExpectedInput>Valid data row</ExpectedInput>
		///		<ExpectedOutput>Valid data row</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1688"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Loads a data row from an object
		/// </summary>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.BloodUnitTestGuid))
			{
				dtRow[TABLE.BloodUnitTest.BloodUnitTestGuid] = this.BloodUnitTestGuid;
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.BloodUnitGuid))
			{
				dtRow[TABLE.BloodUnit.BloodUnitGuid] = this.BloodUnitGuid;
			}
			
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.EntryMethodCode))
			{
				dtRow[TABLE.BloodUnitTest.EntryMethodCode] = this.EntryMethodCode;
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.AntiseraWorklistTypingGuid))
			{
				if (this._antiseraWorklistTypingGuid != System.Guid.Empty)
				{
					dtRow[TABLE.BloodUnitTest.AntiseraWorklistTypingGuid] = this._antiseraWorklistTypingGuid;
				}
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.ConfirmationWorklistUnitGuid))
			{
				if (this._confirmationWorklistUnitGuid != System.Guid.Empty)
				{
					dtRow[TABLE.BloodUnitTest.ConfirmationWorklistUnitGuid] = this._confirmationWorklistUnitGuid;
				}
			}

			if(dtRow.Table.Columns.Contains(TABLE.AntiseraWorklistTyping.OrderableTestId))
			{
				dtRow[TABLE.AntiseraWorklistTyping.OrderableTestId] = this._orderableTestId;
			}

			if(dtRow.Table.Columns.Contains(TABLE.Reagent.ReagentGuid))
			{
				if (this._reagent.ReagentGuid != System.Guid.Empty)
				{
					dtRow[TABLE.Reagent.ReagentGuid] = this._reagent.ReagentGuid;
				}
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.RowVersion))
			{
				dtRow[TABLE.BloodUnitTest.RowVersion] = this.RowVersion;
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.AutoInstrumentName))
			{
				dtRow[TABLE.BloodUnitTest.AutoInstrumentName] = this._autoInstrumentName;
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.RackGuid))
			{
				if (this.Rack != null)
				{
					dtRow[TABLE.BloodUnitTest.RackGuid] = this.Rack.RackGuid;
				}
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.TestingMethodCode))
			{
				if (this.TestingMethod != null)
				{
					dtRow[TABLE.BloodUnitTest.TestingMethodCode] = this.TestingMethod.TestingMethodCode;
				}
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.TestTechId))
			{
				if (this.TestTechId != string.Empty)
				{
					dtRow[TABLE.BloodUnitTest.TestTechId] = this.TestTechId;
				}
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.EntryTechId))
			{
				if (this._entryTechId != string.Empty)
				{
					dtRow[TABLE.BloodUnitTest.EntryTechId] = this._entryTechId;
				}
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.RecordStatusCode))
			{
				dtRow[TABLE.BloodUnitTest.RecordStatusCode] = this.RecordStatusCode;
			}

			base.LoadDataRowFromThis(dtRow);
			return dtRow;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		/// <summary>
		/// Loads an object from a data row
		/// </summary>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.BloodUnitGuid))
			{
				this.BloodUnitGuid = dtRow.IsNull(TABLE.BloodUnit.BloodUnitGuid) ? System.Guid.Empty : (System.Guid) dtRow[TABLE.BloodUnit.BloodUnitGuid];
				if (this.BloodUnitGuid != Guid.Empty)
				{
					this.BloodUnit = new BOL.BloodUnit(this.BloodUnitGuid);
				}
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.EntryMethodCode))
			{
				this.EntryMethodCode = dtRow.IsNull(TABLE.BloodUnitTest.EntryMethodCode) ? string.Empty : (string) dtRow[TABLE.BloodUnitTest.EntryMethodCode];				
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.AntiseraWorklistTypingGuid))
			{
				this.AntiseraWorklistTypingGuid = dtRow.IsNull(TABLE.BloodUnitTest.AntiseraWorklistTypingGuid) ? Guid.Empty : (System.Guid) dtRow[TABLE.BloodUnitTest.AntiseraWorklistTypingGuid];
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.ConfirmationWorklistUnitGuid))
			{
				this.ConfirmationWorklistUnitGuid = dtRow.IsNull(TABLE.BloodUnitTest.ConfirmationWorklistUnitGuid) ? Guid.Empty : (System.Guid) dtRow[TABLE.BloodUnitTest.ConfirmationWorklistUnitGuid];
			}

			if(dtRow.Table.Columns.Contains(TABLE.AntiseraWorklistTyping.OrderableTestId))
			{
				this.OrderableTestId = dtRow.IsNull(TABLE.AntiseraWorklistTyping.OrderableTestId) ? 0 : (int) dtRow[TABLE.AntiseraWorklistTyping.OrderableTestId];
			}

			if(dtRow.Table.Columns.Contains(TABLE.Reagent.ReagentGuid))
			{
				this.Reagent = dtRow.IsNull(TABLE.Reagent.ReagentGuid) ? (new BOL.Reagent()) : (new BOL.Reagent((System.Guid) dtRow[TABLE.Reagent.ReagentGuid]));
			}

			if(dtRow.Table.Columns.Contains(TABLE.OrderableTest.OrderableTestName))
			{
				this.Specificity = dtRow.IsNull(TABLE.OrderableTest.OrderableTestName) ? String.Empty : (dtRow[TABLE.OrderableTest.OrderableTestName].ToString().Substring(2));
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.RowVersion))
			{
				this.RowVersion = dtRow.IsNull(TABLE.BloodUnitTest.RowVersion) ? new byte[0] : ((byte[])dtRow[TABLE.BloodUnitTest.RowVersion]);
			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.TestTechId))
			{
				this.TestTechId = dtRow[TABLE.BloodUnitTest.TestTechId].ToString();
			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.TestDate))
			{
				this.TestDate = dtRow.IsNull(TABLE.BloodUnitTest.TestDate) ? DateTime.MinValue : Convert.ToDateTime(dtRow[TABLE.BloodUnitTest.TestDate]);
			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.TestResultId))
			{
				this.TestResult = new BOL.TestResult(dtRow[TABLE.BloodUnitTest.TestResultId].ToString());
			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.TestingMethodCode))
			{
				if (!dtRow.IsNull(TABLE.BloodUnitTest.TestingMethodCode))
                    this.TestingMethod = new BOL.TestingMethod(dtRow[TABLE.BloodUnitTest.TestingMethodCode].ToString());
			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.AutoInstrumentName))
			{
				if (!dtRow.IsNull(TABLE.BloodUnitTest.AutoInstrumentName))
				{
					this.AutoInstrumentName = dtRow[TABLE.BloodUnitTest.AutoInstrumentName].ToString();
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.RackGuid))
			{
				if (dtRow.IsNull(TABLE.BloodUnitTest.RackGuid))
				{
					this.Rack = new BOL.Rack();
				}
				else
				{
					this.Rack = new BOL.Rack((Guid)dtRow[TABLE.BloodUnitTest.RackGuid]);
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.OrderedUnitGuid))
			{
				if (dtRow.IsNull(TABLE.BloodUnitTest.OrderedUnitGuid))
				{
					this._orderedUnitGuid = Guid.Empty;
				}
				else
				{
					this._orderedUnitGuid = (Guid)dtRow[TABLE.BloodUnitTest.OrderedUnitGuid];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitTest.EntryTechId))
			{
				if (!dtRow.IsNull(TABLE.BloodUnitTest.EntryTechId))
				{
					this._entryTechId = dtRow[TABLE.BloodUnitTest.EntryTechId].ToString();
				}
			}

            //Defect 251282 Added UnitResultSetGuid to allow differentation between tests for automated instrument tests
            //tests coming from auto instrument don't have the worklist guid assigned to them
            if (dtRow.Table.Columns.Contains(TABLE.PendingBloodUnitTest.UnitResultSetGuid))
            {
                this.UnitResultSetGuid = dtRow.IsNull(TABLE.PendingBloodUnitTest.UnitResultSetGuid) ? Guid.Empty : (System.Guid)dtRow[TABLE.PendingBloodUnitTest.UnitResultSetGuid];
            } 
            base.LoadFromDataRow(dtRow);
		}

		#endregion

		#region Get Methods




		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/4/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7355"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7356"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets an empty table layout for a UC_02's Confirmation DataGrid
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetEmptyAboRhConfirmationTable()
		{
			return(DAL.BloodUnitTest.GetEmptyAboRhConfirmationTable());
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="542"> 
		///		<ExpectedInput>Blood unit Guid, blood test type id, and division code. </ExpectedInput>
		///		<ExpectedOutput>Data table of blood unit test results.</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1969"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Call data access layer to return data table of blood unit test results for given blood unit and test type id.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="bloodTestTypeId"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetBloodUnitTestResult(Guid bloodUnitGuid, int bloodTestTypeId, string divisionCode)
		{
			return DAL.BloodUnitTest.GetBloodUnitTestResult(bloodUnitGuid, bloodTestTypeId, divisionCode);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1689"> 
		///		<ExpectedInput>Blood unit Guid, blood test type id, and division code. </ExpectedInput>
		///		<ExpectedOutput>String</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1690"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>EmptyString</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetBloodUnitTestResultValue
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="bloodTestTypeId"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static string GetBloodUnitTestResultValue(Guid bloodUnitGuid, int bloodTestTypeId, string divisionCode)
		{
			System.Data.DataTable _dtResults = DAL.BloodUnitTest.GetBloodUnitTestResult(bloodUnitGuid, bloodTestTypeId, divisionCode);

			if (_dtResults.Rows.Count == 1)
			{
				return _dtResults.Rows[0][TABLE.BloodUnitTest.TestResultId].ToString();
			}
			else
			{
				return String.Empty;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1691"> 
		///		<ExpectedInput>BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1692"> 
		///		<ExpectedInput>New Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetsBloodUnitSpecialTest by bloodUnitGuid
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static DataTable GetBloodUnitSpecialTest(Guid bloodUnitGuid)
		{
			return DAL.BloodUnitTest.GetBloodUnitSpecialTest(bloodUnitGuid);
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1970"> 
		///		<ExpectedInput>Division code</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1971"> 
		///		<ExpectedInput>Invalid DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Call data access layer to return data table of sickle cell test results for given division.
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetSickleCellNegativeUnitsForDivision(string divisionCode)
		{
			return DAL.OutgoingShipment.GetBloodUnitTestResultsForDivision(divisionCode,(int)Common.TestType.SickleCell, "N");
		}
		
		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1972"> 
		///		<ExpectedInput>Division code</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1973"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Call data access layer to return data table of CMV negative test results for given division.
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetCmvNegativeUnitsForDivision(string divisionCode)
		{
			return DAL.OutgoingShipment.GetBloodUnitTestResultsForDivision(divisionCode, (int)Common.TestType.CMV, "N");
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1693"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1694"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetAboConfirmationLookup
		/// </summary>
		/// <returns></returns>
		public static DataTable GetAboConfirmationLookup()
		{

			return DAL.BloodUnitTest.GetAboConfirmationLookup();

		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1695"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1696"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetAboConfirmationLookup
		/// </summary>
		/// <returns></returns>
		public static DataTable GetRHConfirmationLookup()
		{

			return DAL.BloodUnitTest.GetRHConfirmationLookup();

		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///
		///<TestCases>
		///
		///<Case type="0" testid ="1968"> 
		///		<ExpectedInput>Division code</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1998"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///	</TestCases>
		///	
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Calls the data access layer to get all the blood unit tests for a division.
		///</summary>
		public static DataTable GetAllUnitTestsForDivision(string divisionCode)
		{
			return DAL.OutgoingShipment.GetBloodUnitTestResultsForDivision(divisionCode);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1697"> 
		///		<ExpectedInput>BloodUnitGuid, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1698"> 
		///		<ExpectedInput>New Guid, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns a data table of BloodUnitTests, CR 2899
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetBloodUnitTests(Guid bloodUnitGuid, string divisionCode)
		{
			return BOL.BloodUnitTest.GetBloodUnitTests(bloodUnitGuid.ToString(), divisionCode);
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/29/2010</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9149"> 
		///		<ExpectedInput>BloodUnit string, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable with blood unit test results</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="9150"> 
		///		<ExpectedInput>empty string, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns a data table of BloodUnitTests, CR 2899
		/// </summary>
		/// <param name="bloodUnitGuids"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetBloodUnitTests(string bloodUnitGuids, string divisionCode)
		{
			return DAL.BloodUnitTest.GetBloodUnitTestResults(bloodUnitGuids, divisionCode);
		}



		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1701"> 
		///		<ExpectedInput>boolean</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1702"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Get empty table schema for use when inserting/updating blood unit tests into the database
		/// </summary>
		/// <param name="isUpdate">Update Indicator. True updating</param>
		/// <returns>Blood unit test table schema</returns>
		public static DataTable GetBloodUnitTestTableSchema(bool isUpdate)
		{
			DataTable dt = new DataTable(TABLE.BloodUnitTest.TableName);
			
			dt.Columns.Add(TABLE.BloodUnitTest.TestResultId, typeof(string));
			dt.Columns.Add(TABLE.BloodUnitTest.TestComments, typeof(string));
			dt.Columns.Add(TABLE.BloodUnitTest.TestDate, typeof(DateTime));
			dt.Columns.Add(TABLE.BloodUnitTest.TestTechId, typeof(string));
			dt.Columns.Add(TABLE.BloodUnitTest.EntryTechId, typeof(string));
			dt.Columns.Add(TABLE.BloodUnitTest.DivisionCode, typeof(string));
			dt.Columns.Add(TABLE.BloodUnitTest.BloodUnitTestGuid, typeof(Guid));

			if (isUpdate)
			{
				dt.Columns.Add(TABLE.BloodUnitTest.RowVersion, typeof(byte[]));
			}
			else
			{
				dt.Columns.Add(TABLE.BloodUnitTest.AntiseraWorklistTypingGuid, typeof(Guid));
				dt.Columns.Add(TABLE.BloodUnitTest.BloodUnitGuid, typeof(Guid));
				dt.Columns.Add(TABLE.BloodUnitTest.BloodTestTypeId, typeof(int));
				dt.Columns.Add(TABLE.BloodUnitTest.EntryMethodCode, typeof(string));
				dt.Columns.Add(TABLE.BloodUnitTest.TestingMethodCode, typeof(char));
				dt.Columns.Add(TABLE.BloodUnitTest.AutoInstrumentName, typeof(string));
				dt.Columns.Add(TABLE.BloodUnitTest.RackGuid, typeof(Guid));
			}
			
			return dt;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1703"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1704"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetBloodUnitTestTableForXM
		/// </summary>
		/// <returns></returns>
		public static DataTable GetBloodUnitTestTableForXM(bool update)
		{
			DataTable dt = null;

			if (update == false)
			{
				dt = new DataTable("BloodUnitTestInsert");
				dt.Columns.Add(TABLE.BloodUnitTest.BloodUnitGuid, typeof(Guid));
				dt.Columns.Add(TABLE.BloodUnitTest.BloodTestTypeId, typeof(int));
				dt.Columns.Add(TABLE.BloodUnitTest.EntryMethodCode, typeof(string));
				dt.Columns.Add(TABLE.BloodUnitTest.OrderedComponentGuid, typeof(Guid));
				dt.Columns.Add(TABLE.BloodUnitTest.OrderedTestGuid, typeof(Guid));
				dt.Columns.Add(TABLE.BloodUnitTest.TestingMethodCode, typeof(char));
				dt.Columns.Add(TABLE.BloodUnitTest.CorrectedResultIndicator, typeof(bool));
				dt.Columns.Add(TABLE.BloodUnitTest.RackGuid, typeof(Guid));
				dt.Columns.Add(TABLE.BloodUnitTest.AutoInstrumentName);
				dt.Columns.Add(TABLE.BloodUnitTest.ReactivityPhaseCode, typeof(char));
				dt.Columns.Add(TABLE.BloodUnitTest.OrderedUnitGuid, typeof(Guid));
			}
			else
			{
				dt = new DataTable("BloodUnitTestUpdate");
				dt.Columns.Add(TABLE.BloodUnitTest.RowVersion, typeof(byte[]));
			}
			
			dt.Columns.Add(TABLE.BloodUnitTest.BloodUnitTestGuid, typeof(Guid));
			dt.Columns.Add(TABLE.BloodUnitTest.TestResultId, typeof(string));
			dt.Columns.Add(TABLE.BloodUnitTest.TestComments, typeof(string));
			dt.Columns.Add(TABLE.BloodUnitTest.TestDate, typeof(DateTime));
			dt.Columns.Add(TABLE.BloodUnitTest.DivisionCode, typeof(string));
			dt.Columns.Add(TABLE.BloodUnitTest.TestTechId, typeof(string));
			dt.Columns.Add(TABLE.BloodUnitTest.EntryTechId, typeof(string));
			dt.Columns.Add(TABLE.BloodUnitTest.LastUpdateFunctionId, typeof(int));

			return dt;
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3754"> 
		///		<ExpectedInput>Valid blood unit Guid and division code</ExpectedInput>
		///		<ExpectedOutput>An array list containing ABO/Rh Confirmation tests</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3755"> 
		///		<ExpectedInput>Invalid blood unit Guid and division code</ExpectedInput>
		///		<ExpectedOutput>An empty array list</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves a list of all Abo/Rh Confirmation tests for a specified blood unit
		/// Implements BR_70.05
		/// </summary>
		/// <param name="bloodUnitGuid">The blood unit Guid</param>
		/// <param name="divisionCode">The division code of the unit's division</param>
		/// <returns>An ArrayList of Abo/Rh confirmation tests</returns>
		public static ArrayList GetAboRHConfirmationTests(System.Guid bloodUnitGuid, string divisionCode)
		{
			//BR_70.05
			DataTable dtTests = DAL.BloodUnitTest.GetAboRHConfirmationTests(bloodUnitGuid, divisionCode);
			ArrayList alTests = new ArrayList();
			foreach (DataRow dr in dtTests.Rows)
			{
				BloodUnitTest buTest = new BloodUnitTest(dr);
				alTests.Add(buTest);
			}
			return alTests;
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3756"> 
		///		<ExpectedInput>Valid blood unit Guid and division code</ExpectedInput>
		///		<ExpectedOutput>An array list containing Antigen typing tests</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3757"> 
		///		<ExpectedInput>Invalid blood unit Guid and division code</ExpectedInput>
		///		<ExpectedOutput>An empty array list</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves a list of all Antigen typing tests for a specified blood unit
		/// Implements BR_70.01, BR_70.04, BR_70.05
		/// </summary>
		/// <param name="bloodUnitGuid">The blood unit Guid</param>
		/// <param name="divisionCode">The division code of the unit's division</param>
		/// <returns>An ArrayList of Abo/Rh confirmation tests</returns>
		public static ArrayList GetAntigenTypingTests(System.Guid bloodUnitGuid, string divisionCode)
		{
			//BR_70.05
			DataTable dtTests = DAL.BloodUnitTest.GetAntigenTypingTests(bloodUnitGuid, divisionCode);
			ArrayList alTests = new ArrayList();
			foreach (DataRow dr in dtTests.Rows)
			{
				BloodUnitTest buTest = new BloodUnitTest(dr);
				alTests.Add(buTest);
			}
			return alTests;
		}



		#endregion

		#region Update/Save Methods

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="543"> 
		///		<ExpectedInput>Valid blood test Guid and test result id.</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2000"> 
		///		<ExpectedInput>Invalid RowVersion</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<param name="lastUpdateFunctionId">UC or calling method</param>
		/// <summary>
		/// Public method to call data access layer to update test result of an exisiting blood test
		/// </summary>
		public void UpdateTestResult(Common.UpdateFunction lastUpdateFunctionId)
		{
			DAL.BloodUnitTest.UpdateTestResult(this.BloodUnitTestGuid,this.TestResultId, this.TestDate, this.TestTechId, this.RowVersion, lastUpdateFunctionId);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="849"> 
		///		<ExpectedInput>WorklistGuid, UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2014"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<param name="confirmationWorklistUnitGuid">Worklist Identifier</param>
		///<param name="lastUpdateFunctionId">UC or calling method</param>
		/// <summary>
		/// Public method which calls data access layer to save new blood unit test to file
		/// </summary>
		public void SaveBloodUnitTest(System.Guid confirmationWorklistUnitGuid, Common.UpdateFunction lastUpdateFunctionId)
		{
			DAL.BloodUnitTest.SaveBloodUnitTest(this.BloodUnitGuid,this.BloodTestTypeObj.BloodTestTypeId,this.TestResultId,this.EntryMethodCode,this.TestComments,Common.LogonUser.LogonUserDivisionCode, lastUpdateFunctionId, confirmationWorklistUnitGuid);
		}


		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/18/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="423"> 
		///		<ExpectedInput>Valid data table, Date, Strings</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2018"> 
		///		<ExpectedInput>Invalid BloodUnit Tests</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayer Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Saves all ABO/Rh blood unit tests, Implements BR_56.07
		/// </summary>
		/// <param name="bloodUnitTests"></param>
		/// <param name="testDateTime"></param>
		/// <param name="testUser"></param>
		/// <param name="automatedInstrName"></param>
		/// <param name="automatedInstrQCd"></param>
		/// <param name="rackGuid"></param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <param name="workloadProcessId"></param>
		/// <param name="rackNotQcComment">Used to implement BR_56.07</param>
		public static void SaveAboRHBloodUnitTests(DataTable bloodUnitTests, DateTime testDateTime, string testUser, string automatedInstrName, bool automatedInstrQCd, Guid rackGuid, Common.UpdateFunction lastUpdateFunctionId, Common.WorkloadProcessID workloadProcessId, string rackNotQcComment)
		{
			DataTable dtExceptionUnitAboRh = null;
			DataTable dtExceptionNotQc = null;
			DataTable dtWorkloadEvents = null;
			//CR 2425, only diminision the array to as big as the rows are valid.
			int nWorkloadRecords = 0;
			foreach(DataRow drTest in bloodUnitTests.Rows)
			{
				if (drTest[0].ToString() == "V" || drTest[0].ToString() == "W")
				{
					nWorkloadRecords++;
				}
			}
			System.Guid [] bloodUnitGuids = new System.Guid[nWorkloadRecords];
			BOL.Rack tmpRack = new BOL.Rack(rackGuid);

			int recCtr = 0;

			foreach(DataRow drTest in bloodUnitTests.Rows)
			{
				if (drTest[0].ToString() == "V" || drTest[0].ToString() == "W")
				{
					bloodUnitGuids[recCtr] = (System.Guid) drTest[TABLE.BloodUnit.BloodUnitGuid];
					if (!drTest.IsNull(ARTIFICIAL.BR221))
					{
						BOL.ExceptionReport tmpReport221 = (BOL.ExceptionReport)drTest[ARTIFICIAL.BR221];
						if (dtExceptionUnitAboRh == null)
						{
							dtExceptionUnitAboRh = tmpReport221.ExceptionData.Table.Clone();
						}
                        //*** Fortify Justified Code ***
                        //dtExceptionUnitAboRh can't be null since it is checked earler (see above). 
                        dtExceptionUnitAboRh.Rows.Add(tmpReport221.ExceptionData.ItemArray);
                        //*** Fortify Justified Code ***
                    }

                    if (!drTest.IsNull(ARTIFICIAL.BR217))
					{
						BOL.ExceptionReport tmpReport217 = (BOL.ExceptionReport)drTest[ARTIFICIAL.BR217];
						if (dtExceptionUnitAboRh == null)
						{
							dtExceptionUnitAboRh = tmpReport217.ExceptionData.Table.Clone();
						}
                        //*** Fortify Justified Code ***
                        //dtExceptionUnitAboRh can't be null since it is checked earler (see above). 
                        dtExceptionUnitAboRh.Rows.Add(tmpReport217.ExceptionData.ItemArray);
                        //*** Fortify Justified Code ***
                    }

                    if (rackNotQcComment != string.Empty)
					{
						BOL.ExceptionReport tmpReport5607 = new BOL.ExceptionReport(Common.ExceptionType.QC_not_performed_on_rack_used_for_testing, rackNotQcComment, Common.LogonUser.LogonUserName);
						tmpReport5607.GenerateExceptionQCNotPerformed(drTest[TABLE.BloodUnit.EyeReadableUnitId].ToString(), drTest[TABLE.BloodProduct.ProductCode].ToString(), tmpRack.RackName, testUser);
						if (dtExceptionNotQc == null)
						{
							dtExceptionNotQc = tmpReport5607.ExceptionData.Table.Clone();
						}
                        //*** Fortify Justified Code ***
                        //dtExceptionNotQc can't be null since it is checked earler (see above). 
                        dtExceptionNotQc.Rows.Add(tmpReport5607.ExceptionData.ItemArray);
                        //*** Fortify Justified Code ***
                    }

                    recCtr++;
				}
			}

		    if (workloadProcessId != Common.WorkloadProcessID.Unknown)
		    {
		        //CR3558 - double workload was being generated
		        if (dtWorkloadEvents == null)
		        {
		            dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData((Common.WorkloadProcessID) workloadProcessId, bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
		        }
		    }


		    DAL.BloodUnitTest.SaveAboRHBloodUnitTests(bloodUnitTests, dtExceptionUnitAboRh, dtWorkloadEvents, testDateTime, testUser, automatedInstrName, automatedInstrQCd, rackGuid, Common.LogonUser.LogonUserDivisionCode, lastUpdateFunctionId, dtExceptionNotQc);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1705"> 
		///		<ExpectedInput>Valid AGC antigen typing test results on an issued unit - for AHG CC phases</ExpectedInput>
		///		<ExpectedOutput>Success indicator of true</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1706"> 
		///		<ExpectedInput>Invalid AGE antigen typing test results - bad row version</ExpectedInput>
		///		<ExpectedOutput>Success indicator of false</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="7039"> 
		///		<ExpectedInput>Valid AGc antigen typing test results on a unit - for IS RT phases</ExpectedInput>
		///		<ExpectedOutput>Success indicator of true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="7040"> 
		///		<ExpectedInput>Valid AGe antigen typing test results on a unit - for IS 37 phases</ExpectedInput>
		///		<ExpectedOutput>Success indicator of true</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Saves Antigen Typing Blood Unit Tests
		/// Implements BR_41.06, BR_41.12, BR_41.16, BR_41.22, BR_56.29
		/// </summary>
		/// <param name="antiseraWorklistTypingGuid"></param>
		/// <param name="antiseraWorklist"></param>
		/// <param name="bloodUnitTests"></param>
		/// <param name="qcTests"></param>
		/// <param name="reactivityPhase"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="workloadProcessIds">Process to associate workload</param>
		/// <param name="bloodUnitGuids"></param>
		/// <param name="partiallyTested"></param>
		/// <returns></returns>
		public static bool SaveAntigenTypingBloodUnitTests(System.Guid antiseraWorklistTypingGuid, 
			BOL.AntiseraWorklist antiseraWorklist,
			DataTable bloodUnitTests, 
			DataTable qcTests,
			string reactivityPhase,
			Common.UpdateFunction lastUpdateFunctionId,
			ArrayList workloadProcessIds,
			System.Guid [] bloodUnitGuids,
			bool partiallyTested)
		{
			DataTable bloodUnitTestTable = BOL.BloodUnitTest.GetBloodUnitTestTableSchema(true);
			//
			DataTable rackTestTable = BOL.RackTest.GetRackTestTableSchema(false);
			System.Data.DataTable dtWorkloadEvents = null;
			System.Data.DataTable dtWorkloadTemp = null;
			//
			DataTable qcRackLotTable = BOL.RackLot.GetRackLotTableSchema(false);
			DataTable qcNewRackLotTable = BOL.RackLot.GetRackLotTableSchema(true);
			//
			DataTable qcAntiseraUnitTestTable = BOL.AntiseraWorklist.GetAntiseraUnitTestTableSchema();
			//
			DataTable dtExceptionExpiredReagentQc = null;
			DataTable dtExceptionIssuedUnitTest = null;
			//
			foreach (DataRow dr in bloodUnitTests.Rows)
			{
				//Quarantine Unit?
				//BR_41.22 & BR_41.12
				//
				if ((bool)dr[TABLE.BloodUnitStatus.QuarantineIndicator])
				{
					BOL.BloodUnit unit = new BOL.BloodUnit((Guid)dr[TABLE.BloodUnit.BloodUnitGuid]);
					BOL.DiscardOrQuarantineBloodUnit quarantine = new BOL.DiscardOrQuarantineBloodUnit();
					quarantine.BloodUnits.Add(unit);
					//
					quarantine.CommentText = "Other";
					quarantine.Details = Common.StrRes.OtherMsg.UC41.UnitAntigenTypeMismatch().ResString;
					//
					BOL.ExceptionReport exReport = new BOL.ExceptionReport(Common.ExceptionType.Unit_quarantined,Common.StrRes.OtherMsg.UC41.UnitAntigenTypeMismatch().ResString,Common.LogonUser.LogonUserName);
					exReport.GenerateExceptionQuarantine(unit);
					ArrayList exceptions = new ArrayList(1);
					exceptions.Add(exReport);
					//
					
					quarantine.ChangeQuarantineStatus(exceptions, true, lastUpdateFunctionId, workloadProcessIds);
				}
				//
				//Patient Antigen Typing mismatch
				//BR_56.29
				if (!dr.IsNull(TABLE.ExceptionQc.ExceptionComment))
				{
					DataRow drOrderableTest = BOL.ReferenceData.GetOrderableTest(Convert.ToInt32(dr[TABLE.OrderableTest.OrderableTestId]));
					DataTable issuedUnit = DAL.IssuedUnit.GetIssuedUnitByBloodUnitGuid((Guid)dr[TABLE.BloodUnit.BloodUnitGuid]);
					BOL.ExceptionReport exReport = new BOL.ExceptionReport(Common.ExceptionType.Unit_ER_issued_testing_problem,dr[TABLE.ExceptionQc.ExceptionComment].ToString(),Common.LogonUser.LogonUserName);
					exReport.GenerateExceptionEmergencyIssueUnitTest((Guid)dr[TABLE.BloodUnit.BloodUnitGuid],(Guid)issuedUnit.Rows[0][TABLE.Patient.PatientGuid],drOrderableTest[TABLE.OrderableTest.OrderableTestText].ToString(),dr[ARTIFICIAL.Interp].ToString(),antiseraWorklist.WorklistName);
					if (exReport.ExceptionData != null)
					{
						if (dtExceptionIssuedUnitTest == null)
						{
							dtExceptionIssuedUnitTest = exReport.ExceptionData.Table.Clone();
						}
                        //*** Fortify Justified Code ***
                        //dtExceptionIssuedUnitTest can't be null since it is checked earler (see above). 
                        dtExceptionIssuedUnitTest.Rows.Add(exReport.ExceptionData.ItemArray);
                        //*** Fortify Justified Code ***
                    }
                }
				//
				//IS
				//	
				if ((dr[ARTIFICIAL.IS].ToString().Length>0)&&((bool)dr[ARTIFICIAL.ISSave]))
				{
					DataRow drAdd = bloodUnitTestTable.NewRow();
					drAdd[TABLE.BloodUnitTest.TestResultId] = dr[ARTIFICIAL.IS];
					drAdd[TABLE.BloodUnitTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.BloodUnitTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					drAdd[TABLE.BloodUnitTest.BloodUnitTestGuid] = dr[ARTIFICIAL.ISGuid];
					drAdd[TABLE.BloodUnitTest.RowVersion] = dr[ARTIFICIAL.ISRowVersion];
					drAdd[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					bloodUnitTestTable.Rows.Add(drAdd);
				}
				//
				//RT
				//
				if ((dr[ARTIFICIAL.RT].ToString().Length>0)&&((bool)dr[ARTIFICIAL.RTSave]))
				{
					DataRow drAdd = bloodUnitTestTable.NewRow();
					drAdd[TABLE.BloodUnitTest.TestResultId] = dr[ARTIFICIAL.RT];
					drAdd[TABLE.BloodUnitTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.BloodUnitTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					drAdd[TABLE.BloodUnitTest.BloodUnitTestGuid] = dr[ARTIFICIAL.RTGuid];
					drAdd[TABLE.BloodUnitTest.RowVersion] = dr[ARTIFICIAL.RTRowVersion];
					drAdd[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					bloodUnitTestTable.Rows.Add(drAdd);
				}
				//
				//37C
				//
				if ((dr[ARTIFICIAL.C37].ToString().Length>0)&&((bool)dr[ARTIFICIAL.C37Save]))
				{
					DataRow drAdd = bloodUnitTestTable.NewRow();
					drAdd[TABLE.BloodUnitTest.TestResultId] = dr[ARTIFICIAL.C37];
					drAdd[TABLE.BloodUnitTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.BloodUnitTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					drAdd[TABLE.BloodUnitTest.BloodUnitTestGuid] = dr[ARTIFICIAL.C37Guid];
					drAdd[TABLE.BloodUnitTest.RowVersion] = dr[ARTIFICIAL.C37RowVersion];
					drAdd[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					bloodUnitTestTable.Rows.Add(drAdd);
				}
				//
				//AHG
				//
				if ((dr[ARTIFICIAL.AHG].ToString().Length>0)&&((bool)dr[ARTIFICIAL.AHGSave]))
				{
					DataRow drAdd = bloodUnitTestTable.NewRow();
					drAdd[TABLE.BloodUnitTest.TestResultId] = dr[ARTIFICIAL.AHG];
					drAdd[TABLE.BloodUnitTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.BloodUnitTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					drAdd[TABLE.BloodUnitTest.BloodUnitTestGuid] = dr[ARTIFICIAL.AHGGuid];
					drAdd[TABLE.BloodUnitTest.RowVersion] = dr[ARTIFICIAL.AHGRowVersion];
					drAdd[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					bloodUnitTestTable.Rows.Add(drAdd);
				}
				//
				//CC
				//
				if ((dr[ARTIFICIAL.CC].ToString().Length>0)&&((bool)dr[ARTIFICIAL.CCSave]))
				{
					DataRow drAdd = bloodUnitTestTable.NewRow();
					drAdd[TABLE.BloodUnitTest.TestResultId] = dr[ARTIFICIAL.CC];
					drAdd[TABLE.BloodUnitTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.BloodUnitTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					drAdd[TABLE.BloodUnitTest.BloodUnitTestGuid] = dr[ARTIFICIAL.CCGuid];
					drAdd[TABLE.BloodUnitTest.RowVersion] = dr[ARTIFICIAL.CCRowVersion];
					drAdd[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					bloodUnitTestTable.Rows.Add(drAdd);
				}
				//
				//Interp
				//
				if ((dr[ARTIFICIAL.Interp].ToString().Length>0)&&((bool)dr[ARTIFICIAL.InterpSave]))
				{
					DataRow drAdd = bloodUnitTestTable.NewRow();
					drAdd[TABLE.BloodUnitTest.TestResultId] = dr[ARTIFICIAL.Interp];
					drAdd[TABLE.BloodUnitTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.BloodUnitTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					drAdd[TABLE.BloodUnitTest.BloodUnitTestGuid] = dr[ARTIFICIAL.InterpGuid];
					drAdd[TABLE.BloodUnitTest.TestComments] = dr[ARTIFICIAL.Comment];
					drAdd[TABLE.BloodUnitTest.RowVersion] = dr[ARTIFICIAL.InterpRowVersion];
					drAdd[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					bloodUnitTestTable.Rows.Add(drAdd);
				}
			}
			//
			if (qcTests.Rows.Count>1)
			{
				//
				//POS Control
				//
				//OverrideComment = Exception for expired pos ctrl reagent
				//
				DataRow drPosControl = qcTests.Rows[0];
				//
				if (drPosControl[ARTIFICIAL.OverrideComment].ToString().Trim().Length>0)
				{
					BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.Expired_antisera_used, drPosControl[ARTIFICIAL.OverrideComment].ToString(), Common.LogonUser.LogonUserName);
					Guid posControlGuid = new Guid(drPosControl[ARTIFICIAL.ControlReagentGuid].ToString());
					//
					//Need to generate an exception for each blood unit involved
					for (int idx = 0; idx < bloodUnitGuids.Length; idx++)
					{
						exRep.GenerateExceptionQC(posControlGuid, Guid.Empty, bloodUnitGuids[idx], Guid.Empty, "", "", Common.Utility.GetReactivityPhaseTextFromCode(reactivityPhase), antiseraWorklist.WorklistName);
					}
					//
					if (exRep.ExceptionData != null)
					{
						if (dtExceptionExpiredReagentQc == null)
						{
							dtExceptionExpiredReagentQc = exRep.ExceptionData.Table.Clone();
						}
						dtExceptionExpiredReagentQc.Rows.Add(exRep.ExceptionData.ItemArray);
					}
				}
				//
				if ((bool)drPosControl[ARTIFICIAL.PartiallyCompleted] == false)
				{
					//insert new rack lot for Pos control QC
					DataRow drPos = qcNewRackLotTable.NewRow();
					drPos[TABLE.RackLot.RackLotGuid] = Guid.NewGuid();
					drPos[TABLE.RackLot.ReagentGuid] = drPosControl[ARTIFICIAL.ControlReagentGuid];
					drPos[TABLE.RackLot.LotVialNumber] = drPosControl[ARTIFICIAL.VialId]; 
					drPos[TABLE.RackLot.LotUseDate] = antiseraWorklist.WorklistCreateDate;
					drPos[TABLE.RackLot.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drPos[TABLE.RackLot.ControlCellId] = true;
					qcNewRackLotTable.Rows.Add(drPos);
					
					DataRow drPosAUT = qcAntiseraUnitTestTable.NewRow();
					drPosAUT[TABLE.AntiseraUnitTest.AntiseraUnitTestGuid] = Guid.NewGuid();
					drPosAUT[TABLE.BloodUnitTest.AntiseraWorklistTypingGuid] = antiseraWorklistTypingGuid;
					drPosAUT[TABLE.AntiseraUnitTest.RackLotGuid] = drPos[TABLE.RackLot.RackLotGuid];
					qcAntiseraUnitTestTable.Rows.Add(drPosAUT);
				}
				//Update interp for Pos QC
				if ((bool)drPosControl[ARTIFICIAL.ControlInterpSave])
				{
					DataRow drRackLot = qcRackLotTable.NewRow();
					drRackLot[TABLE.RackLot.RackLotGuid] = drPosControl[ARTIFICIAL.ControlInterpGuid];
					if (drPosControl[ARTIFICIAL.ControlInterp].ToString().Length>0)
					{
						if (drPosControl[ARTIFICIAL.ControlInterp].ToString()=="P")
						{
							drRackLot[TABLE.RackLot.SatisfactoryResultIndicator] = true;
						}
						else if (drPosControl[ARTIFICIAL.ControlInterp].ToString()=="I")
						{
							drRackLot[TABLE.RackLot.SatisfactoryResultIndicator] = false;
						}
					}
					drRackLot[TABLE.RackLot.Comments] = drPosControl[ARTIFICIAL.ControlComment];
					drRackLot[TABLE.RackLot.ReagentGuid] = drPosControl[ARTIFICIAL.ControlReagentGuid];
					drRackLot[TABLE.RackLot.LotUseDate] = antiseraWorklist.WorklistCreateDate;
					drRackLot[TABLE.RackLot.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drRackLot[TABLE.RackLot.RowVersion] = drPosControl[ARTIFICIAL.InterpRowVersion];
					drRackLot[TABLE.RackLot.TestWithId] = drPosControl[TABLE.RackLot.TestWithId];
					qcRackLotTable.Rows.Add(drRackLot);
				}
				//
				//NEG Control
				//
				//OverrideComment = Exception for expired neg ctrl reagent
				//
				DataRow drNegControl = qcTests.Rows[1];
				//
				if (drNegControl[ARTIFICIAL.OverrideComment].ToString().Trim().Length>0)
				{
					BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.Expired_antisera_used, drNegControl[ARTIFICIAL.OverrideComment].ToString(), Common.LogonUser.LogonUserName);
					Guid negControlGuid = new Guid(drNegControl[ARTIFICIAL.ControlReagentGuid].ToString());
					//
					//Need to generate an exception for each blood unit involved
					for (int idx = 0; idx < bloodUnitGuids.Length; idx++)
					{
						exRep.GenerateExceptionQC(negControlGuid, Guid.Empty, bloodUnitGuids[idx], Guid.Empty, "", "", Common.Utility.GetReactivityPhaseTextFromCode(reactivityPhase), antiseraWorklist.WorklistName);
					}
					//
					if (exRep.ExceptionData != null)
					{
						if (dtExceptionExpiredReagentQc == null)
						{
							dtExceptionExpiredReagentQc = exRep.ExceptionData.Table.Clone();
						}
                        //*** Fortify Justified Code ***
                        //dtExceptionExpiredReagentQc can't be null since it is checked earler (see above). 
                        dtExceptionExpiredReagentQc.Rows.Add(exRep.ExceptionData.ItemArray);
                        //*** Fortify Justified Code ***
                    }
                }
				if ((bool)drNegControl[ARTIFICIAL.PartiallyCompleted] == false)
				{
					//insert new rack lot for Neg control QC
					DataRow drNeg = qcNewRackLotTable.NewRow();
					drNeg[TABLE.RackLot.RackLotGuid] = Guid.NewGuid();
					drNeg[TABLE.RackLot.ReagentGuid] = drNegControl[ARTIFICIAL.ControlReagentGuid];
					drNeg[TABLE.RackLot.LotVialNumber] = drNegControl[ARTIFICIAL.VialId];
					drNeg[TABLE.RackLot.LotUseDate] = antiseraWorklist.WorklistCreateDate;
					drNeg[TABLE.RackLot.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drNeg[TABLE.RackLot.ControlCellId] = false;
					qcNewRackLotTable.Rows.Add(drNeg);
					//
					DataRow drNegAUT = qcAntiseraUnitTestTable.NewRow();
					drNegAUT[TABLE.AntiseraUnitTest.AntiseraUnitTestGuid] = Guid.NewGuid();
					drNegAUT[TABLE.BloodUnitTest.AntiseraWorklistTypingGuid] = antiseraWorklistTypingGuid;
					drNegAUT[TABLE.AntiseraUnitTest.RackLotGuid] = drNeg[TABLE.RackLot.RackLotGuid];
					qcAntiseraUnitTestTable.Rows.Add(drNegAUT);
				}
				//Just update interp for QC
				if ((bool)drNegControl[ARTIFICIAL.ControlInterpSave])
				{
					DataRow drRackLot = qcRackLotTable.NewRow();
					drRackLot[TABLE.RackLot.RackLotGuid] = drNegControl[ARTIFICIAL.ControlInterpGuid];
					if (drNegControl[ARTIFICIAL.ControlInterp].ToString().Length>0)
					{
						if (drNegControl[ARTIFICIAL.ControlInterp].ToString()=="N")
						{
							drRackLot[TABLE.RackLot.SatisfactoryResultIndicator] = true;
						}
						else if (drNegControl[ARTIFICIAL.ControlInterp].ToString()=="I")
						{
							drRackLot[TABLE.RackLot.SatisfactoryResultIndicator] = false;
						}
					}
					drRackLot[TABLE.RackLot.Comments] = drNegControl[ARTIFICIAL.ControlComment];
					drRackLot[TABLE.RackLot.ReagentGuid] = drNegControl[ARTIFICIAL.ControlReagentGuid];
					drRackLot[TABLE.RackLot.LotUseDate] = antiseraWorklist.WorklistCreateDate;
					drRackLot[TABLE.RackLot.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drRackLot[TABLE.RackLot.RowVersion] = drNegControl[ARTIFICIAL.InterpRowVersion];
					drRackLot[TABLE.RackLot.TestWithId] = drNegControl[TABLE.RackLot.TestWithId];
					qcRackLotTable.Rows.Add(drRackLot);
				}
			}
			//
			foreach (DataRow dr in qcTests.Rows)
			{
				//
				//IS
				//	
				if ((dr[ARTIFICIAL.IS].ToString().Length>0)&&((bool)dr[ARTIFICIAL.ISSave]))
				{
					DataRow drAdd = rackTestTable.NewRow();
					drAdd[TABLE.RackTest.RackTestResult] = dr[ARTIFICIAL.IS];
					drAdd[TABLE.RackTest.RackTestGuid] = dr[ARTIFICIAL.ISGuid];
					drAdd[TABLE.RackTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.RackTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.RackTest.RowVersion] = dr[ARTIFICIAL.ISRowVersion];
					drAdd[TABLE.RackTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					rackTestTable.Rows.Add(drAdd);
				}
				//
				//RT
				//
				if ((dr[ARTIFICIAL.RT].ToString().Length>0)&&((bool)dr[ARTIFICIAL.RTSave]))
				{
					DataRow drAdd = rackTestTable.NewRow();
					drAdd[TABLE.RackTest.RackTestResult] = dr[ARTIFICIAL.RT];
					drAdd[TABLE.RackTest.RackTestGuid] = dr[ARTIFICIAL.RTGuid];
					drAdd[TABLE.RackTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.RackTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.RackTest.RowVersion] = dr[ARTIFICIAL.RTRowVersion];
					drAdd[TABLE.RackTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					rackTestTable.Rows.Add(drAdd);
				}
				//
				//37C
				//
				if ((dr[ARTIFICIAL.C37].ToString().Length>0)&&((bool)dr[ARTIFICIAL.C37Save]))
				{
					DataRow drAdd = rackTestTable.NewRow();
					drAdd[TABLE.RackTest.RackTestResult] = dr[ARTIFICIAL.C37];
					drAdd[TABLE.RackTest.RackTestGuid] = dr[ARTIFICIAL.C37Guid];
					drAdd[TABLE.RackTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.RackTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.RackTest.RowVersion] = dr[ARTIFICIAL.C37RowVersion];
					drAdd[TABLE.RackTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					rackTestTable.Rows.Add(drAdd);
				}
				//
				//AHG
				//
				if ((dr[ARTIFICIAL.AHG].ToString().Length>0)&&((bool)dr[ARTIFICIAL.AHGSave]))
				{
					DataRow drAdd = rackTestTable.NewRow();
					drAdd[TABLE.RackTest.RackTestResult] = dr[ARTIFICIAL.AHG];
					drAdd[TABLE.RackTest.RackTestGuid] = dr[ARTIFICIAL.AHGGuid];
					drAdd[TABLE.RackTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.RackTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.RackTest.RowVersion] = dr[ARTIFICIAL.AHGRowVersion];
					drAdd[TABLE.RackTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					rackTestTable.Rows.Add(drAdd);
				}
				//
				//CC
				//
				if ((dr[ARTIFICIAL.CC].ToString().Length>0)&&((bool)dr[ARTIFICIAL.CCSave]))
				{
					DataRow drAdd = rackTestTable.NewRow();
					drAdd[TABLE.RackTest.RackTestResult] = dr[ARTIFICIAL.CC];
					drAdd[TABLE.RackTest.RackTestGuid] = dr[ARTIFICIAL.CCGuid];
					drAdd[TABLE.RackTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.RackTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.RackTest.RowVersion] = dr[ARTIFICIAL.CCRowVersion];
					drAdd[TABLE.RackTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					rackTestTable.Rows.Add(drAdd);
				}
			}
		
			
			if(!partiallyTested)
			{
				for (int i=0; i<workloadProcessIds.Count; i++)
				{
					if (dtWorkloadEvents == null)
					{
						dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData((Common.WorkloadProcessID)workloadProcessIds[i], bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
					}
					else
					{
						dtWorkloadTemp = BOL.WorkloadEvent.GenerateWorkloadData((Common.WorkloadProcessID)workloadProcessIds[i], bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
						if (dtWorkloadTemp.Rows.Count > 0)
						{
                            //*** Fortify Justified Code ***
                            //dtWorkloadEvents can't be null since it is checked earler (see above). 
                            dtWorkloadEvents.Rows.Add(dtWorkloadTemp.Rows[0].ItemArray);
                            //*** Fortify Justified Code ***
                        }
                    }
				}
			}
			//
			return DAL.BloodUnitTest.SaveAntigenTypingBloodUnitTests(dtWorkloadEvents, bloodUnitTestTable,rackTestTable,qcNewRackLotTable,qcRackLotTable,qcAntiseraUnitTestTable, dtExceptionExpiredReagentQc, dtExceptionIssuedUnitTest, lastUpdateFunctionId);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1708"> 
		///		<ExpectedInput>Guid, worklist, Tests, BloodUnits</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1709"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayer Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8472"> 
		///		<ExpectedInput>Issued Unit</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="9140"> 
		///		<ExpectedInput>Issued Unit, blank Weak D Interp</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Saves Weak D Typing Blood Unit Tests
		/// Implements BR_41.16
		/// </summary>
		/// <param name="antiseraWorklistTypingGuid"></param>
		/// <param name="antiseraWorklist"></param>
		/// <param name="weakDTests"></param>
		/// <param name="lastUpdateFunctionId">UC or calling Method</param>
		/// <param name="workloadProcessIDs"></param>
		/// <param name="bloodUnitGuids"></param>
		/// <returns></returns>
		public static bool SaveWeakDBloodUnitTests(Guid antiseraWorklistTypingGuid,
			BOL.AntiseraWorklist antiseraWorklist,
			DataTable weakDTests, 
			Common.UpdateFunction lastUpdateFunctionId,
			ArrayList workloadProcessIDs,
			System.Guid [] bloodUnitGuids)
		{
			DataTable bloodUnitTestTable = BOL.BloodUnitTest.GetBloodUnitTestTableSchema(true);
			//
			DataTable dtExceptionIssuedUnitTest = null;
			System.Data.DataTable dtWorkloadEvents = null;
			System.Data.DataTable dtWorkloadTemp = null;
			//
			foreach (DataRow dr in weakDTests.Rows)
			{
				//Quarantine Unit?
				//BR_41.22
				//
				if ((bool)dr[TABLE.BloodUnitStatus.QuarantineIndicator])
				{
					BOL.BloodUnit unit = new BOL.BloodUnit((Guid)dr[TABLE.BloodUnit.BloodUnitGuid]);
					BOL.DiscardOrQuarantineBloodUnit quarantine = new BOL.DiscardOrQuarantineBloodUnit();
					quarantine.BloodUnits.Add(unit);
					//
					quarantine.CommentText = "Other";
					//CR 2856 Blood Unit status comment is different if the interpretation was inconclusive
					if (dr[ARTIFICIAL.Interp].ToString() == "I")
						quarantine.Details = "Inconclusive Weak D test on Rh Negative unit";
					else if (dr[ARTIFICIAL.Interp].ToString() == "P")
						quarantine.Details = "Positive Weak D test on Rh Negative unit";
					else
						quarantine.Details = "";
					quarantine.ChangeQuarantineStatus(new ArrayList(), true, lastUpdateFunctionId, workloadProcessIDs);
				}
				//
				//Patient Antigen Typing mismatch
				//BR_56.29
				if (!dr.IsNull(TABLE.ExceptionQc.ExceptionComment))
				{
					DataRow drOrderableTest = BOL.ReferenceData.GetOrderableTest(Convert.ToInt32(dr[TABLE.OrderableTest.OrderableTestId]));
					DataTable issuedUnit = DAL.IssuedUnit.GetIssuedUnitByBloodUnitGuid((Guid)dr[TABLE.BloodUnit.BloodUnitGuid]);
					BOL.ExceptionReport exReport = new BOL.ExceptionReport(Common.ExceptionType.Unit_ER_issued_testing_problem,dr[TABLE.ExceptionQc.ExceptionComment].ToString(),Common.LogonUser.LogonUserName);
					exReport.GenerateExceptionEmergencyIssueUnitTest((Guid)dr[TABLE.BloodUnit.BloodUnitGuid],(Guid)issuedUnit.Rows[0][TABLE.Patient.PatientGuid],drOrderableTest[TABLE.OrderableTest.OrderableTestText].ToString(),dr[ARTIFICIAL.Interp].ToString(),antiseraWorklist.WorklistName);
					if (exReport.ExceptionData != null)
					{
						if (dtExceptionIssuedUnitTest == null)
						{
							dtExceptionIssuedUnitTest = exReport.ExceptionData.Table.Clone();
						}
                        //*** Fortify Justified Code ***
                        //dtExceptionIssuedUnitTest can't be null since it is checked earler (see above). 
                        dtExceptionIssuedUnitTest.Rows.Add(exReport.ExceptionData.ItemArray);
                        //*** Fortify Justified Code ***
                    }
                }
				//
				//Anti-D 37
				//	
				if ((dr[ARTIFICIAL.D37].ToString().Length>0)&&((bool)dr[ARTIFICIAL.D37Save]))
				{
					DataRow drAdd = bloodUnitTestTable.NewRow();
					drAdd[TABLE.BloodUnitTest.TestResultId] = dr[ARTIFICIAL.D37];
					drAdd[TABLE.BloodUnitTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.BloodUnitTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					drAdd[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drAdd[TABLE.BloodUnitTest.BloodUnitTestGuid] = dr[ARTIFICIAL.D37Guid];
					drAdd[TABLE.BloodUnitTest.RowVersion] = dr[ARTIFICIAL.D37RowVersion];
					bloodUnitTestTable.Rows.Add(drAdd);
				}
				//
				//Anti-D Control 37
				//
				if ((dr[ARTIFICIAL.Dc37].ToString().Length>0)&&((bool)dr[ARTIFICIAL.Dc37Save]))
				{
					DataRow drAdd = bloodUnitTestTable.NewRow();
					drAdd[TABLE.BloodUnitTest.TestResultId] = dr[ARTIFICIAL.Dc37];
					drAdd[TABLE.BloodUnitTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.BloodUnitTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					drAdd[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drAdd[TABLE.BloodUnitTest.BloodUnitTestGuid] = dr[ARTIFICIAL.Dc37Guid];
					drAdd[TABLE.BloodUnitTest.RowVersion] = dr[ARTIFICIAL.Dc37RowVersion];
					bloodUnitTestTable.Rows.Add(drAdd);
				}
				//
				//Anti-D AHG
				//
				if ((dr[ARTIFICIAL.DAHG].ToString().Length>0)&&((bool)dr[ARTIFICIAL.DAHGSave]))
				{
					DataRow drAdd = bloodUnitTestTable.NewRow();
					drAdd[TABLE.BloodUnitTest.TestResultId] = dr[ARTIFICIAL.DAHG];
					drAdd[TABLE.BloodUnitTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.BloodUnitTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					drAdd[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drAdd[TABLE.BloodUnitTest.BloodUnitTestGuid] = dr[ARTIFICIAL.DAHGGuid];
					drAdd[TABLE.BloodUnitTest.RowVersion] = dr[ARTIFICIAL.DAHGRowVersion];
					bloodUnitTestTable.Rows.Add(drAdd);
				}
				//
				//Anti-D Control AHG
				//
				if ((dr[ARTIFICIAL.DcAHG].ToString().Length>0)&&((bool)dr[ARTIFICIAL.DcAHGSave]))
				{
					DataRow drAdd = bloodUnitTestTable.NewRow();
					drAdd[TABLE.BloodUnitTest.TestResultId] = dr[ARTIFICIAL.DcAHG];
					drAdd[TABLE.BloodUnitTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.BloodUnitTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					drAdd[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drAdd[TABLE.BloodUnitTest.BloodUnitTestGuid] = dr[ARTIFICIAL.DcAHGGuid];
					drAdd[TABLE.BloodUnitTest.RowVersion] = dr[ARTIFICIAL.DcAHGRowVersion];
					bloodUnitTestTable.Rows.Add(drAdd);
				}
				//
				//Anti-D CC
				//
				if ((dr[ARTIFICIAL.DCC].ToString().Length>0)&&((bool)dr[ARTIFICIAL.DCCSave]))
				{
					DataRow drAdd = bloodUnitTestTable.NewRow();
					drAdd[TABLE.BloodUnitTest.TestResultId] = dr[ARTIFICIAL.DCC];
					drAdd[TABLE.BloodUnitTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.BloodUnitTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					drAdd[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drAdd[TABLE.BloodUnitTest.BloodUnitTestGuid] = dr[ARTIFICIAL.DCCGuid];
					drAdd[TABLE.BloodUnitTest.RowVersion] = dr[ARTIFICIAL.DCCRowVersion];
					bloodUnitTestTable.Rows.Add(drAdd);
				}
				//
				//Anti-D Control CC
				//
				if ((dr[ARTIFICIAL.DcCC].ToString().Length>0)&&((bool)dr[ARTIFICIAL.DcCCSave]))
				{
					DataRow drAdd = bloodUnitTestTable.NewRow();
					drAdd[TABLE.BloodUnitTest.TestResultId] = dr[ARTIFICIAL.DcCC];
					drAdd[TABLE.BloodUnitTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.BloodUnitTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					drAdd[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drAdd[TABLE.BloodUnitTest.BloodUnitTestGuid] = dr[ARTIFICIAL.DcCCGuid];
					drAdd[TABLE.BloodUnitTest.RowVersion] = dr[ARTIFICIAL.DcCCRowVersion];
					bloodUnitTestTable.Rows.Add(drAdd);
				}
				//
				//Interp
				//
				if ((dr[ARTIFICIAL.Interp].ToString().Length>0)&&((bool)dr[ARTIFICIAL.InterpSave]))
				{
					DataRow drAdd = bloodUnitTestTable.NewRow();
					drAdd[TABLE.BloodUnitTest.TestResultId] = dr[ARTIFICIAL.Interp];
					drAdd[TABLE.BloodUnitTest.TestDate] = antiseraWorklist.WorklistCreateDate;
					drAdd[TABLE.BloodUnitTest.TestTechId] = antiseraWorklist.WorklistName;
					drAdd[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
					drAdd[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drAdd[TABLE.BloodUnitTest.BloodUnitTestGuid] = dr[ARTIFICIAL.InterpGuid];
					drAdd[TABLE.BloodUnitTest.RowVersion] = dr[ARTIFICIAL.InterpRowVersion];
					drAdd[TABLE.BloodUnitTest.TestComments] = dr[ARTIFICIAL.Comment];
					bloodUnitTestTable.Rows.Add(drAdd);
				}
			}
			for (int i=0; i<workloadProcessIDs.Count; i++)
			{
				if (dtWorkloadEvents == null)
				{
					if((Common.WorkloadProcessID)workloadProcessIDs[i] == Common.WorkloadProcessID.AntigenPhenotypingMultipleTestPhasesQC || (Common.WorkloadProcessID)workloadProcessIDs[i] == Common.WorkloadProcessID.AntigenPhenotypingSingleTestPhaseQC)
					{
						dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData((Common.WorkloadProcessID)workloadProcessIDs[i], bloodUnitGuids, Common.WorkloadTransactionType.Miscellanious, false);
					}
					else
					{
						dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData((Common.WorkloadProcessID)workloadProcessIDs[i], bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
					}
				}
				else
				{
					if((Common.WorkloadProcessID)workloadProcessIDs[i] == Common.WorkloadProcessID.AntigenPhenotypingMultipleTestPhasesQC || (Common.WorkloadProcessID)workloadProcessIDs[i] == Common.WorkloadProcessID.AntigenPhenotypingSingleTestPhaseQC)
					{
						dtWorkloadTemp = BOL.WorkloadEvent.GenerateWorkloadData((Common.WorkloadProcessID)workloadProcessIDs[i], bloodUnitGuids, Common.WorkloadTransactionType.Miscellanious, false);
					}
					else
					{
						dtWorkloadTemp = BOL.WorkloadEvent.GenerateWorkloadData((Common.WorkloadProcessID)workloadProcessIDs[i], bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
					}
                    
                    //CR 3554
                    if (dtWorkloadTemp == null || dtWorkloadEvents == null)
                        throw new ApplicationException("dtWorkloadEvents or dtWorkloadTemp are null");
                    //*** Fortify Justified Code ***
                    //dtWorkloadEvents can't be null since it is checked earler (see above). 
                    if (dtWorkloadTemp.Rows.Count > 0) dtWorkloadEvents.Rows.Add(dtWorkloadTemp.Rows[0].ItemArray);
                    //*** Fortify Justified Code ***
                }
            }

			return DAL.BloodUnitTest.SaveWeakDBloodUnitTests(bloodUnitTestTable, dtWorkloadEvents, dtExceptionIssuedUnitTest, lastUpdateFunctionId);
		}
        


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3810"> 
		///		<ExpectedInput>Crossmatch grid row</ExpectedInput>
		///		<ExpectedOutput>BloodUnitTest row</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1710"> 
		///		<ExpectedInput>Invalid crossmatch grid row</ExpectedInput>
		///		<ExpectedOutput>Exception is thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// BuildBloodUnitTestRowsFromXM
		/// BR_40.14
		/// </summary>
		/// <param name="dtXMResults"></param>
		/// <param name="dtBloodUnitTestsInsert"></param>
		/// <param name="dtBloodUnitTestsUpdate"></param>
		/// <param name="dtOrderedUnits"></param>
		/// <param name="testingDateTime"></param>
		/// <param name="testingTechId"></param>
		/// <param name="reactivityPhase"></param>
		/// <param name="divisionCode"></param>
		/// <param name="rackGuid"></param>
		/// <param name="automatedInstrumentName"></param>
		/// <param name="lastUpdateFunctionId"></param>
		public static void BuildBloodUnitTestRowsFromXM(DataTable dtXMResults, 
			DataTable dtBloodUnitTestsInsert,
			DataTable dtBloodUnitTestsUpdate,
			DataTable dtOrderedUnits,
			DateTime testingDateTime,
			string testingTechId,
			Common.ReactivityPhase reactivityPhase,
			string divisionCode, 
			Guid rackGuid, 
			string automatedInstrumentName,
			Common.UpdateFunction lastUpdateFunctionId)
		{
			DataTable dtBloodTestTypes = BOL.BloodTestType.GetBloodTestTypes(dtXMResults.Rows[0][TABLE.BloodTestType.OrderableTestId].ToString());
			DataRow drOrderedUnit = null;
			bool releaseUnit = false;

			foreach(DataRow drXMResult in dtXMResults.Rows)
			{
				foreach(DataColumn col in dtXMResults.Columns)
				{
					DataRow drBloodUnitTest = null;
					bool update = false;

					if (drXMResult.IsNull(col) == false)
					{
						if (col.ColumnName == ARTIFICIAL.IS || col.ColumnName == ARTIFICIAL.C37 || col.ColumnName == ARTIFICIAL.AHG ||
							col.ColumnName == ARTIFICIAL.CC || col.ColumnName == ARTIFICIAL.XMInterp)
						{
							string saveColName = col.ColumnName + "Save";
							string guidColName = col.ColumnName + ARTIFICIAL.Guid;
							string rowVersionColName = col.ColumnName + TABLE.BloodUnit.RowVersion;

							if ((bool)drXMResult[saveColName] == false)
								continue;

							if (drXMResult.IsNull(guidColName) == true)
							{
								drBloodUnitTest = dtBloodUnitTestsInsert.NewRow();
								drBloodUnitTest[TABLE.BloodUnitTest.BloodUnitTestGuid] = System.Guid.NewGuid();
								dtBloodUnitTestsInsert.Rows.Add(drBloodUnitTest);
								update = false;
							}
							else
							{
								drBloodUnitTest = dtBloodUnitTestsUpdate.NewRow();
								drBloodUnitTest[TABLE.BloodUnitTest.BloodUnitTestGuid] = drXMResult[guidColName];
								drBloodUnitTest[TABLE.BloodUnitTest.RowVersion] = drXMResult[rowVersionColName];
								dtBloodUnitTestsUpdate.Rows.Add(drBloodUnitTest);
								update = true;
							}

							if (drXMResult[col].ToString().Trim() != "")
							{
								drBloodUnitTest[TABLE.BloodUnitTest.TestResultId] = drXMResult[col];
							}
							else
							{
								drBloodUnitTest[TABLE.BloodUnitTest.TestResultId] = DBNull.Value;
							}

							if (col.ColumnName == ARTIFICIAL.IS)
							{
								if (update == false)
									drBloodUnitTest[TABLE.BloodUnitTest.BloodTestTypeId] = dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '" + "*IS*" + "'")[0][TABLE.BloodUnitTest.BloodTestTypeId];
							}
							else if (col.ColumnName == ARTIFICIAL.C37)
							{
								if (update == false)
									drBloodUnitTest[TABLE.BloodUnitTest.BloodTestTypeId] = dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '" + "*37*" + "'")[0][TABLE.BloodUnitTest.BloodTestTypeId];
							}
							else if (col.ColumnName == ARTIFICIAL.AHG)
							{
								if (update == false)
									drBloodUnitTest[TABLE.BloodUnitTest.BloodTestTypeId] = dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '" + "*AHG*" + "'")[0][TABLE.BloodUnitTest.BloodTestTypeId];
							}
							else if (col.ColumnName == ARTIFICIAL.CC)
							{
								if (update == false)
									drBloodUnitTest[TABLE.BloodUnitTest.BloodTestTypeId] = dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '" + "*CC*" + "'")[0][TABLE.BloodUnitTest.BloodTestTypeId];
							}
							else if (col.ColumnName == ARTIFICIAL.XMInterp)
							{
								if (update == false)
									drBloodUnitTest[TABLE.BloodUnitTest.BloodTestTypeId] = dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '" + "*Interp*" + "'")[0][TABLE.BloodUnitTest.BloodTestTypeId];

								if (drXMResult[col].ToString().Trim() != "")
								{
									drOrderedUnit = dtOrderedUnits.NewRow();
									drOrderedUnit[TABLE.OrderedUnit.OrderedUnitGuid] = drXMResult[TABLE.OrderedUnit.OrderedUnitGuid];
									drOrderedUnit[TABLE.OrderedUnit.RowVersion] = drXMResult[ARTIFICIAL.OrderedUnitRowVersion];
									drOrderedUnit[TABLE.OrderedUnit.CrossmatchDate] = testingDateTime;
									drOrderedUnit[TABLE.OrderedUnit.CrossmatchTechId] = testingTechId;
									drOrderedUnit[TABLE.OrderedUnit.CrossmatchResultCode] = drXMResult[col];
									drOrderedUnit[TABLE.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
									drOrderedUnit[TABLE.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
									drOrderedUnit[TABLE.OrderedUnit.RecordStatusCode] = (char)Common.RecordStatusCode.Active;
								
									BOL.BloodUnitStatus bloodUnitStatus = new BOL.BloodUnitStatus((Guid)drXMResult[TABLE.OrderedUnit.BloodUnitGuid]);

									//compatible
									//incompatible - give only with Medical Director approval
									if (drXMResult[col].ToString().Trim() == Common.TestResult.C.ToString() ||
										drXMResult[col].ToString().Trim() == Common.TestResult.G.ToString() ||
										((bloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Issued ||
										 bloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Transfused) &&
										 drXMResult[col].ToString().Trim() != Common.TestResult.Z.ToString()))
									{
										drOrderedUnit[TABLE.OrderedUnit.RecordStatusCode] = System.Convert.ToChar(Common.RecordStatusCode.Active);
									}
										//incompatible
										//compatible - do not transfuse
									else if (drXMResult[col].ToString().Trim() == Common.TestResult.I.ToString() ||
										drXMResult[col].ToString().Trim() == Common.TestResult.D.ToString())
									{
									    if (drOrderedUnit != null) //redundant statement for Fortify
									    {
									        //CR 3554
									        char recordStatusCode = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
									        if (recordStatusCode != '\0')
									            drOrderedUnit[TABLE.OrderedUnit.RecordStatusCode] = recordStatusCode;
									        drOrderedUnit[TABLE.OrderedUnit.ReleaseDate] = testingDateTime;
									        drOrderedUnit[TABLE.OrderedUnit.ReleaseTechId] = testingTechId;
									        releaseUnit = true;
									    }
									}

									dtOrderedUnits.Rows.Add(drOrderedUnit);

									if (drXMResult.IsNull(TABLE.OrderedUnit.OrderedTestGuid) == false)
									{
										DataTable dtOrderedTest = BOL.OrderedTest.GetOrderedTest((Guid)drXMResult[TABLE.OrderedTest.OrderedTestGuid]);
										if (dtOrderedTest.Rows.Count > 0)
										{
											if (dtOrderedTest.Rows[0].IsNull(TABLE.OrderedTest.RelatedComponentGuid) == false)
											{
												DataTable dtOriginalOrderedUnits = BOL.OrderedUnit.GetOrderedUnitByComponent((Guid)dtOrderedTest.Rows[0][TABLE.OrderedTest.RelatedComponentGuid], Common.LogonUser.LogonUserDivisionCode);

												if (dtOriginalOrderedUnits.Select(TABLE.BloodUnit.BloodUnitGuid + " = '" + drXMResult[TABLE.BloodUnit.BloodUnitGuid].ToString() + "'").Length > 0 &&
														 releaseUnit == true)
												{
													DataRow drOriginalOrderedUnit = dtOriginalOrderedUnits.Select(TABLE.BloodUnit.BloodUnitGuid + " = '" + drXMResult[TABLE.BloodUnit.BloodUnitGuid].ToString() + "'")[0];
													drOrderedUnit = dtOrderedUnits.NewRow();
													drOrderedUnit[TABLE.OrderedUnit.OrderedUnitGuid] = drOriginalOrderedUnit[TABLE.OrderedUnit.OrderedUnitGuid];
													drOrderedUnit[TABLE.OrderedUnit.RowVersion] = drOriginalOrderedUnit[TABLE.OrderedUnit.RowVersion];
													drOrderedUnit[TABLE.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
													drOrderedUnit[TABLE.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
													drOrderedUnit[TABLE.OrderedUnit.RecordStatusCode] = (char)Common.RecordStatusCode.Active;
													drOrderedUnit[TABLE.OrderedUnit.ReleaseDate] = testingDateTime;
													drOrderedUnit[TABLE.OrderedUnit.ReleaseTechId] = testingTechId;
													dtOrderedUnits.Rows.Add(drOrderedUnit);
												}
											}
										}
									}
								}
							}

                            if (drBloodUnitTest == null)
                                throw new ApplicationException("drBloodUnitTest is null");

							if (update == false)
							{
                                //CR 3554
							    string entryCode = Common.Utility.GetTestEntryMethodCodeFromTestEntryMethodEnum(Common.TestEntryMethod.Manual);
                                if (entryCode != string.Empty)
								    drBloodUnitTest[TABLE.BloodUnitTest.EntryMethodCode] = System.Convert.ToChar(Common.TestEntryMethod.Manual);
								drBloodUnitTest[TABLE.BloodUnitTest.ReactivityPhaseCode] = ((int)reactivityPhase).ToString()[0];
								drBloodUnitTest[TABLE.BloodUnitTest.BloodUnitGuid] = drXMResult[TABLE.BloodUnit.BloodUnitGuid];
								drBloodUnitTest[TABLE.BloodUnitTest.OrderedTestGuid] = drXMResult[TABLE.OrderedTest.OrderedTestGuid];
								drBloodUnitTest[TABLE.BloodUnitTest.OrderedComponentGuid] = drXMResult[TABLE.OrderedComponent.OrderedComponentGuid];
								drBloodUnitTest[TABLE.BloodUnitTest.CorrectedResultIndicator] = drXMResult[TABLE.BloodUnitTest.CorrectedResultIndicator];
								if (rackGuid != Guid.Empty)
								{
									drBloodUnitTest[TABLE.BloodUnitTest.RackGuid] = rackGuid;
								}
								drBloodUnitTest[TABLE.BloodUnitTest.AutoInstrumentName] = automatedInstrumentName;
								drBloodUnitTest[TABLE.BloodUnitTest.OrderedUnitGuid] = drXMResult[TABLE.BloodUnitTest.OrderedUnitGuid];
							}
                            //CR 3554
                            drBloodUnitTest[TABLE.BloodUnitTest.TestDate] = testingDateTime == System.DateTime.MinValue ? System.DateTime.Now : testingDateTime;
							drBloodUnitTest[TABLE.BloodUnitTest.TestTechId] = testingTechId == String.Empty? String.Empty : testingTechId;
						    string logonUser = Common.LogonUser.LogonUserName;
                            drBloodUnitTest[TABLE.BloodUnitTest.EntryTechId] = logonUser == string.Empty ? String.Empty : logonUser;
							drBloodUnitTest[TABLE.BloodUnitTest.DivisionCode] = divisionCode == string.Empty? String.Empty : divisionCode;
                            string comments = drXMResult[ARTIFICIAL.XMComment].ToString();
                            drBloodUnitTest[TABLE.BloodUnitTest.TestComments] = comments == string.Empty ? String.Empty : comments;
						    int functionId = (int) lastUpdateFunctionId;
                            drBloodUnitTest[TABLE.BloodUnitTest.LastUpdateFunctionId] = functionId == int.MinValue ? int.MinValue : functionId;

						}
					}
				}
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///
		///<TestCases>
		///	
		///<Case type="0" testid ="1711"> 
		///		<ExpectedInput>Crossmatch test</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="6830"> 
		///		<ExpectedInput>RepeatCrossmatch test</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1712"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// BuildXMFromBloodUnitTest
		/// </summary>
		/// <param name="dtXM"></param>
		/// <param name="orderedTestGuid"></param>
		/// <param name="orderedComponentGuid"></param>
		public static void BuildXMFromBloodUnitTest(DataTable dtXM, Guid orderedTestGuid, Guid orderedComponentGuid)
		{
			DataTable dtBloodUnitTests;
			int reactivityPhase = -1; 

			dtBloodUnitTests = DAL.BloodUnitTest.GetTestResultsForXM(orderedTestGuid, orderedComponentGuid);

            // CR 3570: Moved this code so that it's triggered in cases where all tests have been deactivated.
            if (dtBloodUnitTests.Rows.Count > 0)
            {
                reactivityPhase = Convert.ToInt32(dtBloodUnitTests.Rows[0][TABLE.BloodUnitTest.ReactivityPhaseCode].ToString());
            }
            
			foreach(DataRow drXM in dtXM.Rows)
			{
				DataRow[] dtBloodUnitTestsForRow = null;
				if (drXM.IsNull(TABLE.BloodUnitTest.OrderedUnitGuid) == false)
				{
					dtBloodUnitTestsForRow = dtBloodUnitTests.Select(TABLE.BloodUnitTest.OrderedUnitGuid + "= '" + drXM[TABLE.BloodUnitTest.OrderedUnitGuid].ToString() + "'");
				}
				else
				{
					dtBloodUnitTestsForRow = dtBloodUnitTests.Select(TABLE.BloodUnitTest.BloodUnitGuid + "= '" + drXM[TABLE.BloodUnitTest.BloodUnitGuid].ToString() + "'");
				}

				foreach (DataRow drBloodUnitTest in dtBloodUnitTestsForRow)
				{
					if (drBloodUnitTest.IsNull(TABLE.SpecimenTest.CorrectedResultIndicator))
					{
						drXM[TABLE.SpecimenTest.CorrectedResultIndicator] = false;
					}
					else
					{
						drXM[TABLE.SpecimenTest.CorrectedResultIndicator] = (bool)drBloodUnitTest[TABLE.SpecimenTest.CorrectedResultIndicator];
					}

					if (drBloodUnitTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.IS) != -1)
					{
						if (drBloodUnitTest.IsNull(TABLE.BloodUnitTest.TestResultId) == false)
						{
							drXM[ARTIFICIAL.IS] = drBloodUnitTest[TABLE.BloodUnitTest.TestResultId].ToString().Trim();
							drXM[ARTIFICIAL.ISSave] = false;
						}
						else
						{
							drXM[ARTIFICIAL.IS] = "";
							drXM[ARTIFICIAL.ISSave] = true;
						}
						drXM[ARTIFICIAL.ISGuid] = drBloodUnitTest[TABLE.BloodUnitTest.BloodUnitTestGuid];
						drXM[ARTIFICIAL.ISRowVersion] = drBloodUnitTest[TABLE.BloodUnitTest.RowVersion];
					}
					else if (drBloodUnitTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.C37) != -1)
					{
						if (drBloodUnitTest.IsNull(TABLE.BloodUnitTest.TestResultId) == false)
						{
							drXM[ARTIFICIAL.C37] = drBloodUnitTest[TABLE.BloodUnitTest.TestResultId].ToString().Trim();
							drXM[ARTIFICIAL.C37Save] = false;
						}
						else
						{
							drXM[ARTIFICIAL.C37] = "";
							drXM[ARTIFICIAL.C37Save] = true;
						}
						drXM[ARTIFICIAL.C37Guid] = drBloodUnitTest[TABLE.BloodUnitTest.BloodUnitTestGuid];
						drXM[ARTIFICIAL.C37RowVersion] = drBloodUnitTest[TABLE.BloodUnitTest.RowVersion];
					}
					else if (drBloodUnitTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.AHG) != -1)
					{
						if (drBloodUnitTest.IsNull(TABLE.BloodUnitTest.TestResultId) == false)
						{
							drXM[ARTIFICIAL.AHG] = drBloodUnitTest[TABLE.BloodUnitTest.TestResultId].ToString().Trim();
							drXM[ARTIFICIAL.AHGSave] = false;
						}
						else
						{
							drXM[ARTIFICIAL.AHG] = "";
							drXM[ARTIFICIAL.AHGSave] = true;
						}
						drXM[ARTIFICIAL.AHGGuid] = drBloodUnitTest[TABLE.BloodUnitTest.BloodUnitTestGuid];
						drXM[ARTIFICIAL.AHGRowVersion] = drBloodUnitTest[TABLE.BloodUnitTest.RowVersion];
					}
					else if (drBloodUnitTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.CC) != -1)
					{
						if (drBloodUnitTest.IsNull(TABLE.BloodUnitTest.TestResultId) == false)
						{
							drXM[ARTIFICIAL.CC] = drBloodUnitTest[TABLE.BloodUnitTest.TestResultId].ToString().Trim();
							drXM[ARTIFICIAL.CCSave] = false;
						}
						else
						{
							drXM[ARTIFICIAL.CC] = "";
							drXM[ARTIFICIAL.CCSave] = true;
						}
						drXM[ARTIFICIAL.CCGuid] = drBloodUnitTest[TABLE.BloodUnitTest.BloodUnitTestGuid];
						drXM[ARTIFICIAL.CCRowVersion] = drBloodUnitTest[TABLE.BloodUnitTest.RowVersion];
					}
					else if (drBloodUnitTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.Interp) != -1)
					{
						if (drBloodUnitTest.IsNull(TABLE.BloodUnitTest.TestResultId) == false)
						{
							drXM[ARTIFICIAL.XMInterp] = drBloodUnitTest[TABLE.BloodUnitTest.TestResultId].ToString().Trim();
							drXM[ARTIFICIAL.XMInterpSave] = false;
						}
						else
						{
							drXM[ARTIFICIAL.XMInterp] = "";
							drXM[ARTIFICIAL.XMInterpSave] = true;
						}
						drXM[ARTIFICIAL.XMComment] = drBloodUnitTest[TABLE.BloodUnitTest.TestComments].ToString().Trim();
						drXM[ARTIFICIAL.XMInterpGuid] = drBloodUnitTest[TABLE.BloodUnitTest.BloodUnitTestGuid];
						drXM[ARTIFICIAL.XMInterpRowVersion] = drBloodUnitTest[TABLE.BloodUnitTest.RowVersion];		
					}
				}
				if (dtBloodUnitTestsForRow.Length > 0)
				{
					if (dtBloodUnitTestsForRow[0].IsNull(TABLE.Rack.RackGuid) == false)
						drXM[TABLE.Rack.RackGuid] = dtBloodUnitTestsForRow[0][TABLE.Rack.RackGuid];
				}
			}

			if (reactivityPhase == Convert.ToInt32(Common.ReactivityPhase.IAT))
			{
				foreach(DataRow dr in dtXM.Rows)
				{
					dr[ARTIFICIAL.C37] = null;
					dr[ARTIFICIAL.CC] = null;
					dr[ARTIFICIAL.C37Save] = false;
					dr[ARTIFICIAL.CCSave] = false;
                    //CR 3556
                    if (dr.IsNull(ARTIFICIAL.IS) == true)
                    {
                        dr[ARTIFICIAL.IS] = "";
                        dr[ARTIFICIAL.ISSave] = true;
                    }
                    if (dr.IsNull(ARTIFICIAL.AHG) == true)
					{
						dr[ARTIFICIAL.AHG] = "";
						dr[ARTIFICIAL.AHGSave] = true;
					}
				}
			}
			else
			{
				foreach(DataRow dr in dtXM.Rows)
				{
					if (dr.IsNull(ARTIFICIAL.IS) == true)
					{
						dr[ARTIFICIAL.IS] = "";
						dr[ARTIFICIAL.ISSave] = true;
					}
					if (dr.IsNull(ARTIFICIAL.C37) == true)
					{
						dr[ARTIFICIAL.C37] = "";
						dr[ARTIFICIAL.C37Save] = true;
					}
					if (dr.IsNull(ARTIFICIAL.AHG) == true)
					{
						dr[ARTIFICIAL.AHG] = "";
						dr[ARTIFICIAL.AHGSave] = true;
					}
					if (dr.IsNull(ARTIFICIAL.CC) == true)
					{
						dr[ARTIFICIAL.CC] = "";
						dr[ARTIFICIAL.CCSave] = true;
					}
				}
			}
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3750"> 
		///		<ExpectedInput>Valid blood unit guid, confirmation worklist unit guid, comment and last update function</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3751"> 
		///		<ExpectedInput>Invalid blood unit guid, confirmation worklist unit guid, comment and last update function</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update>
		/// 12/6/2004 RDH
		///		Added aboRhConfirmationTests arraylist to the parameter list.
		///		Took the datatable construction from the DAL and moved it to 
		///		this function and populated it with the tests from the 
		///		aboRhConfirmationTests arraylist.
		///</Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inactivate a set of abo/rh confirmation blood unit tests
		/// Implements BR_70.09, BR_70.12, BR_70.25
		/// </summary>
		/// <param name="bloodUnit">The blood unit</param>
		/// <param name="confirmationWorklistUnitGuid">The confirmation worklist unit Guid of tests to be inactivated</param>
		/// <param name="inactivateCommentText">The inactivation comment</param>
		/// <param name="lastUpdateFunctionId">UC or function performing this call</param>
		/// <param name="aboRhConfirmationTests">Tests to inactivate</param>
		/// <param name="allTestsInactivated">Whether all tests for unit are being inactivated</param>
        /// <param name="testingMethod">AutomatedInstrument or Tube</param>
        /// <param name="performAntiD">Was Rh factor tested?</param>
        public static void InactivateAboRHConfirmationTests(
			BOL.BloodUnit bloodUnit, 
			Guid confirmationWorklistUnitGuid, 
			string inactivateCommentText,
			Common.UpdateFunction lastUpdateFunctionId,
			System.Collections.ArrayList aboRhConfirmationTests,
			bool allTestsInactivated,
            string testingMethod,
            bool performAntiD)
		{
			Guid [] arrayBloodUnitGuid = new Guid [1];
			arrayBloodUnitGuid[0] = bloodUnit.BloodUnitGuid;
			DataTable dtWorkload = null;
					
			//
			if (inactivateCommentText.Length == 0)
			{
				throw(new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.UC070.InactivationCommentRequired().ResString));
			}
			//
			DataTable dtTests = new DataTable();
			dtTests.Columns.Add(TABLE.BloodUnit.BloodUnitGuid, typeof(System.Guid));
			dtTests.Columns.Add(TABLE.ConfirmationWorklistUnit.ConfirmationWorklistUnitGuid, typeof(System.Guid));
            dtTests.Columns.Add(TABLE.PendingBloodUnitTest.UnitResultSetGuid, typeof(System.Guid));
            dtTests.Columns.Add(TABLE.BloodUnitMedia.Comments, typeof(string));
			dtTests.Columns.Add(TABLE.BloodUnitTest.InactivateDate, typeof(System.DateTime));
			dtTests.Columns.Add(TABLE.BloodUnitTest.InactivateUser, typeof(string));
			dtTests.Columns.Add(TABLE.BloodUnitTest.DivisionCode, typeof(string));
			dtTests.Columns.Add(TABLE.BloodUnitTest.LastUpdateUser, typeof(string));
			dtTests.Columns.Add(TABLE.BloodUnitTest.LastUpdateFunctionId, typeof(int));
			dtTests.Columns.Add(TABLE.BloodUnitTest.RowVersion, typeof(byte[]));
			//
            Guid uniqueGuid = Guid.Empty;
            
            for (int idx = 0; idx < aboRhConfirmationTests.Count; idx++)
			{
				BOL.BloodUnitTest bloodUnitTest = (BOL.BloodUnitTest)aboRhConfirmationTests[idx];

				uniqueGuid = bloodUnitTest.ConfirmationWorklistUnitGuid == Guid.Empty ? bloodUnitTest.UnitResultSetGuid : bloodUnitTest.ConfirmationWorklistUnitGuid;

                //filtering out unit test results that don't belong to the invalidated test (in case of two or more tests on the list)
                if (uniqueGuid == confirmationWorklistUnitGuid)
                {
                    DataRow dtRow = dtTests.NewRow();
                    dtRow[TABLE.BloodUnit.BloodUnitGuid] = bloodUnit.BloodUnitGuid;

                    //Defect 251282 testing method must be adjusted since it is possible for users to mark manually entered test as done by automated instrument
                    testingMethod = bloodUnitTest.ConfirmationWorklistUnitGuid == Guid.Empty ? Common.TestingMethod.AutomatedInstrument.ToString() : Common.TestingMethod.Tube.ToString();

                    //Defect 251282 - Modified to allow confirmation tests from auto instrument to be invalidated 
                    if (testingMethod == Common.TestingMethod.Tube.ToString())
                    {
                        dtRow[TABLE.ConfirmationWorklistUnit.ConfirmationWorklistUnitGuid] = confirmationWorklistUnitGuid;
                        dtRow[TABLE.PendingBloodUnitTest.UnitResultSetGuid] = DBNull.Value;
                    }
                    else
                    {
                        dtRow[TABLE.ConfirmationWorklistUnit.ConfirmationWorklistUnitGuid] = DBNull.Value;
                        dtRow[TABLE.PendingBloodUnitTest.UnitResultSetGuid] = confirmationWorklistUnitGuid;
                    }

                    dtRow[TABLE.BloodUnitMedia.Comments] = bloodUnitTest.Comments;
                    dtRow[TABLE.BloodUnitTest.InactivateDate] = gov.va.med.vbecs.DAL.VbecsDateTime.GetDivisionCurrentDateTime();
                    dtRow[TABLE.BloodUnitTest.InactivateUser] = Common.LogonUser.LogonUserName;
                    dtRow[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
                    dtRow[TABLE.BloodUnitTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
                    dtRow[TABLE.BloodUnitTest.LastUpdateFunctionId] = (int)lastUpdateFunctionId;
                    dtRow[TABLE.BloodUnitTest.RowVersion] = bloodUnitTest.RowVersion;
                    dtTests.Rows.Add(dtRow);
                }
			}
				
			
			if(performAntiD == false)
			{
                if (testingMethod == Common.TestingMethod.AutomatedInstrument.ToString())
                {
                    dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.AutomatedUnitAbo, arrayBloodUnitGuid, Common.WorkloadTransactionType.Unit, true);
                }
                else
                {
                    dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.ABOForwardTypingUnit, arrayBloodUnitGuid, Common.WorkloadTransactionType.Unit, true);
                }
			}
			else
			{
                if (testingMethod == Common.TestingMethod.AutomatedInstrument.ToString())
                {
                    dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.AutomatedUnitAboRh, arrayBloodUnitGuid, Common.WorkloadTransactionType.Unit, true);
                }
                else
                {
                    dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.ABORhForwardTypingUnit, arrayBloodUnitGuid, Common.WorkloadTransactionType.Unit, true);
                }
			}
		
			dtWorkload = Common.Utility.AppendLastUpdateInformation(dtWorkload, lastUpdateFunctionId);
			// BR_70.09, BR_70.12, BR_70.25
			DAL.BloodUnitTest.InactivateAboRHConfirmationTests( dtTests, dtWorkload );
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3752"> 
		///		<ExpectedInput>Valid blood unit guid, antisera worklist typing guid, orderable test id, comment and last update function</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3753"> 
		///		<ExpectedInput>Invalid blood unit guid, antisera worklist typing guid, orderable test id, comment and last update function</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inactivate a set of antigen typing blood unit tests
		/// Implements BR_70.09, BR_70.12
		/// </summary>
		/// <param name="bloodUnitGuid">The blood unit Guid</param>
		/// <param name="antiseraWorklistTypingGuid">The antisera worklist typing Guid of tests to be inactivated</param>
		/// <param name="inactivateCommentText">The inactivation comment</param>
		/// <param name="lastUpdateFunctionId">UC or function performing this call</param>
		/// <param name="workloadProcessId">Workload process id </param>
		/// <param name="antigenTypingTests">antigenTypingTests</param>
        /// <param name="testingMethod">AutomatedInstrument or Tube</param>
        public static void InactivateAntigenTypingTests
		( 
			System.Guid bloodUnitGuid, 
			System.Guid antiseraWorklistTypingGuid, 
			string inactivateCommentText,
			Common.UpdateFunction lastUpdateFunctionId,
			Common.WorkloadProcessID workloadProcessId,
			ArrayList antigenTypingTests,
            string testingMethod
		)
		{
			if (inactivateCommentText.Length == 0)
			{
				throw(new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.UC070.InactivationCommentRequired().ResString));
			}
			//
			DataTable dtWorkloadEvent = BOL.WorkloadEvent.GenerateWorkloadData
			(
				workloadProcessId, new Guid[]{ bloodUnitGuid }, Common.WorkloadTransactionType.Unit, true
			);
			//
			DataTable dtAntigenTypingTests = new DataTable();
			dtAntigenTypingTests.Columns.Add(TABLE.BloodUnit.BloodUnitGuid, typeof(System.Guid));
			dtAntigenTypingTests.Columns.Add(TABLE.AntiseraWorklistTyping.AntiseraWorklistTypingGuid, typeof(System.Guid));
            dtAntigenTypingTests.Columns.Add(TABLE.PendingBloodUnitTest.UnitResultSetGuid, typeof(System.Guid));
            dtAntigenTypingTests.Columns.Add(TABLE.OrderableTest.OrderableTestId, typeof(int));
			dtAntigenTypingTests.Columns.Add(TABLE.BloodUnitMedia.Comments, typeof(string));
			dtAntigenTypingTests.Columns.Add(TABLE.BloodUnitTest.InactivateDate, typeof(System.DateTime));
			dtAntigenTypingTests.Columns.Add(TABLE.BloodUnitTest.InactivateUser, typeof(string));
			dtAntigenTypingTests.Columns.Add(TABLE.BloodUnitTest.DivisionCode, typeof(string));
			dtAntigenTypingTests.Columns.Add(TABLE.BloodUnitTest.LastUpdateUser, typeof(string));
			dtAntigenTypingTests.Columns.Add(TABLE.BloodUnitTest.LastUpdateFunctionId, typeof(int));
			dtAntigenTypingTests.Columns.Add(TABLE.BloodUnitTest.RowVersion, typeof(byte[]));
			//
			foreach(BOL.BloodUnitTest but in antigenTypingTests)
			{
				if(but.AntiseraWorklistTypingGuid.Equals(antiseraWorklistTypingGuid) ||
                    but.UnitResultSetGuid.Equals(antiseraWorklistTypingGuid))
				{
					DataRow dtRow = dtAntigenTypingTests.NewRow();
					dtRow[TABLE.BloodUnit.BloodUnitGuid] = but.BloodUnitGuid;

                    //Defect 251282 testing method must be adjusted since it is possible for users to mark manually entered test as done by automated instrument
                    testingMethod = but.AntiseraWorklistTypingGuid == Guid.Empty ? Common.TestingMethod.AutomatedInstrument.ToString() : Common.TestingMethod.Tube.ToString();

                    //Defect 251282 - Modified to allow confirmation tests from auto instrument to be invalidated 
                    if (testingMethod == Common.TestingMethod.Tube.ToString())
                    {
                        dtRow[TABLE.AntiseraWorklistTyping.AntiseraWorklistTypingGuid] = antiseraWorklistTypingGuid;
                        dtRow[TABLE.PendingBloodUnitTest.UnitResultSetGuid] = DBNull.Value;
                    }
                    else
                    {
                        dtRow[TABLE.AntiseraWorklistTyping.AntiseraWorklistTypingGuid] = DBNull.Value;
                        dtRow[TABLE.PendingBloodUnitTest.UnitResultSetGuid] = antiseraWorklistTypingGuid;
                    }

					dtRow[TABLE.OrderableTest.OrderableTestId] = but.OrderableTestId;
					dtRow[TABLE.BloodUnitMedia.Comments] = but.Comments;
					dtRow[TABLE.BloodUnitTest.InactivateDate] = gov.va.med.vbecs.DAL.VbecsDateTime.GetDivisionCurrentDateTime();
					dtRow[TABLE.BloodUnitTest.InactivateUser] = Common.LogonUser.LogonUserName;
					dtRow[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					dtRow[TABLE.BloodUnitTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
					dtRow[TABLE.BloodUnitTest.LastUpdateFunctionId] = (int) lastUpdateFunctionId;
					dtRow[TABLE.BloodUnitTest.RowVersion] = but.RowVersion;
					dtAntigenTypingTests.Rows.Add(dtRow);
				}
			}
			//
			int tater = dtAntigenTypingTests.Rows.Count;
			//
			//BR_70.09, BR_70.12
			DAL.BloodUnitTest.InactivateAntigenTypingTests(dtAntigenTypingTests, dtWorkloadEvent, lastUpdateFunctionId);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4119"> 
		///		<ExpectedInput>Valid array list of blood unit test objects, valid ordered unit object, non-empty comment string</ExpectedInput>
		///		<ExpectedOutput>Success indicator of true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4120"> 
		///		<ExpectedInput>Empty comment string</ExpectedInput>
		///		<ExpectedOutput>Business object exception thrown</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8441"> 
		///		<ExpectedInput>Valid array list of blood unit test objects (Repeat), valid ordered unit object, non-empty comment string - repeat XMatch test</ExpectedInput>
		///		<ExpectedOutput>Success indicator of true</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Invalidate completed Crossmatch Test Results, and optionally order re-test
		/// Implements BR_94.04
		/// </summary>
		/// <param name="orderableTest">Orderable test</param>
		/// <param name="bloodUnitTests">Array list of blood unit test objects to invalidate</param>
		/// <param name="orderedUnit">Ordered unit object to remove crossmatch details from</param>
		/// <param name="invalidateComment">Invalidate comment string</param>
		/// <param name="replaceResult">Indicates whether results should be replaced</param>
		/// <param name="alWorkloadProcessIds">workload events</param>
		/// <param name="patientGuid">patient Guid</param>
		/// <param name="lastUpdateFunction">Last update function Id</param>
		/// <returns>Success indicator</returns>
		public static bool InvalidateCrossmatchTestResults(int orderableTest, System.Collections.ArrayList bloodUnitTests, BOL.OrderedUnit orderedUnit, string invalidateComment, bool replaceResult, Guid patientGuid, System.Collections.ArrayList alWorkloadProcessIds, Common.UpdateFunction lastUpdateFunction)
		{
			DataTable dtWorkloadEvent = null;
			if (invalidateComment.Length == 0)
			{
				throw(new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.UC070.InactivationCommentRequired().ResString));
			}
			//
			DateTime currentDateTime = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			//
			DataTable dtBloodUnitTestResults = new DataTable(TABLE.BloodUnitTest.TableName);
			dtBloodUnitTestResults.Columns.Add(TABLE.BloodUnitTest.BloodUnitTestGuid, typeof(Guid));
			dtBloodUnitTestResults.Columns.Add(TABLE.BloodUnitTest.RecordStatusCode, typeof(char));
			dtBloodUnitTestResults.Columns.Add(TABLE.BloodUnitTest.TestComments, typeof(string));
			dtBloodUnitTestResults.Columns.Add(TABLE.BloodUnitTest.InactivateDate, typeof(DateTime));
			dtBloodUnitTestResults.Columns.Add(TABLE.BloodUnitTest.InactivateUser, typeof(string));
			dtBloodUnitTestResults.Columns.Add(TABLE.BloodUnitTest.RowVersion, typeof(byte[]));
			dtBloodUnitTestResults.Columns.Add(TABLE.BloodUnitTest.DivisionCode, typeof(string));
			//
			foreach(BOL.BloodUnitTest test in bloodUnitTests)
			{
				DataRow drBloodUnitTestResults = dtBloodUnitTestResults.NewRow();
				drBloodUnitTestResults[TABLE.BloodUnitTest.BloodUnitTestGuid] = test.BloodUnitTestGuid;
				drBloodUnitTestResults[TABLE.BloodUnitTest.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.ErrorCorrected);
				drBloodUnitTestResults[TABLE.BloodUnitTest.TestComments] = invalidateComment;
				drBloodUnitTestResults[TABLE.BloodUnitTest.InactivateDate] = currentDateTime;
				drBloodUnitTestResults[TABLE.BloodUnitTest.InactivateUser] = Common.LogonUser.LogonUserName;
				drBloodUnitTestResults[TABLE.BloodUnitTest.RowVersion] = test.RowVersion;
				drBloodUnitTestResults[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
				dtBloodUnitTestResults.Rows.Add(drBloodUnitTestResults);
			}
			//
			DataTable dtOrderedUnit = new DataTable(TABLE.OrderedUnit.TableName);
			dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.OrderedUnitGuid, typeof(Guid));
			dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.CrossmatchDate, typeof(DateTime));
			dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.CrossmatchTechId, typeof(string));
			dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.CrossmatchResultCode, typeof(char));
			dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.ReleaseDate, typeof(DateTime));
			dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.ReleaseTechId, typeof(string));
			dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.RecordStatusCode, typeof(char));
			dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.RowVersion, typeof(byte[]));
			dtOrderedUnit.Columns.Add(TABLE.OrderedUnit.DivisionCode, typeof(string));
			//
			DataRow drOrderedUnit = dtOrderedUnit.NewRow();
			drOrderedUnit[TABLE.OrderedUnit.OrderedUnitGuid] = orderedUnit.OrderedUnitGuid;
			drOrderedUnit[TABLE.OrderedUnit.CrossmatchDate] = DBNull.Value;
			drOrderedUnit[TABLE.OrderedUnit.CrossmatchTechId] = DBNull.Value;
			drOrderedUnit[TABLE.OrderedUnit.CrossmatchResultCode] = DBNull.Value;
			if (orderedUnit.ReleaseTechId.ToString().Trim().Length == 0)
			{
				drOrderedUnit[TABLE.OrderedUnit.ReleaseTechId] = DBNull.Value;
			}
			else
			{
				drOrderedUnit[TABLE.OrderedUnit.ReleaseTechId] = orderedUnit.ReleaseTechId;
			}
			if (orderedUnit.ReleaseDate == DateTime.MinValue)
			{
				drOrderedUnit[TABLE.OrderedUnit.ReleaseDate] = DBNull.Value;
			}
			else
			{
				drOrderedUnit[TABLE.OrderedUnit.ReleaseDate] = orderedUnit.ReleaseDate;
			}
			drOrderedUnit[TABLE.OrderedUnit.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(orderedUnit.RecordStatusCode);
			drOrderedUnit[TABLE.OrderedUnit.RowVersion] = orderedUnit.RowVersion;
			drOrderedUnit[TABLE.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtOrderedUnit.Rows.Add(drOrderedUnit);
			//
			DataTable dtOrderedComponent = null;
			DataTable dtOrderedTest = null;
			//
			if ((orderableTest == (int)Common.OrderableTest.XM) && (replaceResult) && (bloodUnitTests.Count>0))
			{
				dtOrderedComponent = DAL.BloodUnitTest.GetBloodUnitTestOrder(((BOL.BloodUnitTest)bloodUnitTests[0]).BloodUnitTestGuid);
				//
				foreach(DataRow drOrderedComponent in dtOrderedComponent.Rows)
				{
					drOrderedComponent[TABLE.OrderedComponent.OrderStatusCode] = Common.Utility.GetOrderStatusCodeFromEnum(Common.OrderStatus.Accepted);
					drOrderedComponent[TABLE.OrderedComponent.TaskStatusCode] = Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.ResultsCorrected);
				}
			}
			else if ((orderableTest == (int)Common.OrderableTest.RepeatXM)&&(bloodUnitTests.Count>0))
			{
				dtOrderedTest = DAL.BloodUnitTest.GetBloodUnitTestOrder(((BOL.BloodUnitTest)bloodUnitTests[0]).BloodUnitTestGuid);
				
				//GetBlodUnitTestOrder returns data for OCs and OTs
				if (dtOrderedTest.Columns.Contains(TABLE.OrderedComponent.EmergencyOrderIndicator))
				{
					dtOrderedTest.Columns.Remove(TABLE.OrderedComponent.EmergencyOrderIndicator);
				}

				if (replaceResult)
				{
					foreach(DataRow drOrderedTest in dtOrderedTest.Rows)
					{
						drOrderedTest[TABLE.OrderedTest.OrderStatusCode] = Common.Utility.GetOrderStatusCodeFromEnum(Common.OrderStatus.Accepted);
						drOrderedTest[TABLE.OrderedTest.TaskStatusCode] = Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.ResultsCorrected);
					}
				}
			}
			//
			Guid [] patientGuids = new Guid[1];
			patientGuids[0] = patientGuid;												
			for(int nCtr = 0; nCtr < alWorkloadProcessIds.Count; nCtr ++)
			{
				System.Data.DataTable dtWorkloadEventTemp = BOL.WorkloadEvent.GenerateWorkloadData((Common.WorkloadProcessID) alWorkloadProcessIds[nCtr], patientGuids, Common.WorkloadTransactionType.Patient, true);
				if(dtWorkloadEvent == null)
				{
					dtWorkloadEvent = dtWorkloadEventTemp.Clone();
				}
				foreach(System.Data.DataRow dtRow in dtWorkloadEventTemp.Rows)
				{
                    //*** Fortify Justified Code ***
                    //dtWorkloadEvent can't be null since it is assigned earler (see above). 
                    DataRow dtNewRow = dtWorkloadEvent.NewRow();
                    //*** Fortify Justified Code ***
                    dtNewRow.ItemArray = dtRow.ItemArray;
					dtWorkloadEvent.Rows.Add(dtNewRow);
				}
			}
			return DAL.BloodUnitTest.InvalidateCrossmatchTestResults(dtBloodUnitTestResults,dtOrderedUnit,dtOrderedTest,dtOrderedComponent,dtWorkloadEvent,lastUpdateFunction);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7408"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7409"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Invalidate partially completed crossmatch results
		/// </summary>
		/// <param name="dtInvalidBloodUnitTest"></param>
		/// <param name="testOrder"></param>
		/// <param name="componentOrder"></param>
		/// <param name="lastUpdateFunction"></param>
		/// <returns></returns>
		public static bool InvalidatePartiallyCompletedCrossmatch(DataTable dtInvalidBloodUnitTest, BOL.OrderedTest testOrder, BOL.OrderedComponent componentOrder, Common.UpdateFunction lastUpdateFunction)
		{
			return DAL.BloodUnitTest.InvalidatePartiallyCompletedCrossmatch(dtInvalidBloodUnitTest);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7412"> 
		///		<ExpectedInput>Valid ordered component guid</ExpectedInput>
		///		<ExpectedOutput>table of XM test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8048"> 
		///		<ExpectedInput>Valid ordered test guid</ExpectedInput>
		///		<ExpectedOutput>table of repeat XM test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7413"> 
		///		<ExpectedInput>Invalid ordered component guid and ordered test guid</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of test results for XM
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <param name="orderedComponentGuid"></param>
		/// <returns></returns>
		public static DataTable GetTestResultsForXM(Guid orderedTestGuid, Guid orderedComponentGuid)
		{
			return DAL.BloodUnitTest.GetTestResultsForXM(orderedTestGuid, orderedComponentGuid);
		}

		#endregion

	}
}



